//
//  TAVMusicSelectorController.m
//  TAVMagicUIKit
//
//  Created by penghuixu on 2024/11/4.
//

#import <UIKit/UIKit.h>
#import <MediaPlayer/MediaPlayer.h>
#import "BgmSelectorView.h"
@import Masonry;

@interface BgmSelectorView () <UITableViewDelegate, UITableViewDataSource>

@property (nonatomic, strong) UITableView *tableView;
@property (nonatomic, strong) NSArray<NSString *> *musicPaths;
@property (nonatomic, strong) NSArray<NSDictionary *> *musicInfos;

@end

@implementation BgmSelectorView

- (instancetype)initWithFrame:(CGRect)frame {
    self = [super initWithFrame:frame];
    if (self) {
        self.backgroundColor = UIColor.whiteColor;
        _musicInfos = @[];
        [self setupTableView];
        // 初始化音乐文件数组
        [self loadMusicFiles];
        [self setupTitleLabel];
        [self setupCloseButton];
        // 初始状态设置为在视图下方
        self.transform = CGAffineTransformMakeTranslation(0, self.bounds.size.height);
        
        // 进入动画
       [UIView animateWithDuration:0.5
                             delay:0
            usingSpringWithDamping:0.9 // 阻尼系数，值越小弹性越强
             initialSpringVelocity:0.3 // 初始速度
                           options:UIViewAnimationOptionCurveEaseInOut
                        animations:^{
           self.transform = CGAffineTransformIdentity; // 恢复到原始位置
       } completion:nil];
    }
    return self;
}

- (void)setupTableView {
//    self.autoresizingMask = UIViewAutoresizingFlexibleHeight;
    self.tableView = [[UITableView alloc] init];
    [self addSubview:self.tableView];
    [self.tableView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.top.equalTo(self.mas_safeAreaLayoutGuideTop).offset(60);
        make.leading.equalTo(self.mas_leading);
        make.trailing.equalTo(self.mas_trailing);
        make.bottom.equalTo(self.mas_bottom);
    }];


    self.tableView.delegate = self;
    self.tableView.dataSource = self;
    [self.tableView registerClass:[UITableViewCell class] forCellReuseIdentifier:@"cell"];
}
- (void)setupTitleLabel {
    UILabel *titleLabel = [[UILabel alloc] init];
//    titleLabel.text = @"选择音乐"; // 设置标题文本
    titleLabel.text =NSLocalizedString(@"Choose a music", "选择音乐");
    titleLabel.font = [UIFont boldSystemFontOfSize:20]; // 设置字体和大小
    titleLabel.textAlignment = NSTextAlignmentCenter; // 设置文本对齐方式
    [self addSubview:titleLabel];

    // 使用 Masonry 设置标题的约束
    [titleLabel mas_makeConstraints:^(MASConstraintMaker *make) {
        if (@available(iOS 11.0, *)) {
            // 使用安全区域布局
            make.top.equalTo(self.mas_safeAreaLayoutGuideTop).offset(0); // 距离安全区顶部20点
        } else {
            // 兼容 iOS 11 以下版本
            make.top.equalTo(self.mas_top).offset(0); // 距离顶部20点
        }
        make.centerX.equalTo(self); // 水平居中
    }];
}


- (void)setupCloseButton {
    UIButton *closeButton = [UIButton buttonWithType:UIButtonTypeSystem];
    [closeButton setTitle:NSLocalizedString(@"Close","关闭") forState:UIControlStateNormal];
    [closeButton addTarget:self action:@selector(closeView) forControlEvents:UIControlEventTouchUpInside];
    
    [self addSubview:closeButton];
    [closeButton mas_makeConstraints:^(MASConstraintMaker *make) {
        if (@available(iOS 11.0, *)) {
            // 使用安全区域布局
            make.top.equalTo(self.mas_safeAreaLayoutGuideTop).offset(0); // 距离顶部20点
        } else {
            // 兼容 iOS 11 以下版本
            make.top.equalTo(self.mas_top).offset(0); // 距离顶部20点
        }
        make.trailing.equalTo(self.mas_trailing).offset(0); // 距离右边20点
        make.width.mas_equalTo(100); // 设置宽度为100
        make.height.mas_equalTo(50); // 设置高度为50
    }];
}
- (void)closeView {
    [UIView animateWithDuration:0.5
                          delay:0
         usingSpringWithDamping:0.9 // 阻尼系数
          initialSpringVelocity:0.3 // 初始速度
                        options:UIViewAnimationOptionCurveEaseInOut
                     animations:^{
        self.transform = CGAffineTransformMakeTranslation(0, self.bounds.size.height); // 移动到视图下方
    } completion:^(BOOL finished) {
        [self removeFromSuperview]; // 动画完成后移除视图
    }];
}

- (void)loadMusicFiles {
    NSMutableArray<NSString *> *musicPaths = [NSMutableArray array];
    NSMutableArray<NSDictionary *> *musicInfos = [NSMutableArray array];

    // 1. 获取 bgm.bundle 的路径
    NSURL *bundleURL = [[NSBundle mainBundle] URLForResource:@"tav_bgm" withExtension:@"bundle"];
    if (!bundleURL) {
        NSLog(@"错误：未找到 bgm.bundle");
        return;
    }
    NSBundle *bgmBundle = [NSBundle bundleWithURL:bundleURL];

    // 2. 读取并解析 bgm.json
    NSString *jsonPath = [bgmBundle pathForResource:@"bgm" ofType:@"json"];
    if (!jsonPath) {
        NSLog(@"错误：bgm.bundle 中未找到 bgm.json");
        return;
    }

    NSData *jsonData = [NSData dataWithContentsOfFile:jsonPath];
    NSError *error = nil;
    NSArray *jsonArray = [NSJSONSerialization JSONObjectWithData:jsonData options:kNilOptions error:&error];

    if (error || ![jsonArray isKindOfClass:[NSArray class]]) {
        NSLog(@"JSON 解析失败: %@", error.localizedDescription);
        return;
    }

    // 3. 遍历 JSON 数据，提取文件路径和信息
    for (NSDictionary *info in jsonArray) {
        NSString *fileName = info[@"fileName"];

        NSString *filePath = [bgmBundle pathForResource:[fileName stringByDeletingPathExtension]
                                                 ofType:[fileName pathExtension]];
        
        if (filePath) {
            [musicPaths addObject:filePath];
            [musicInfos addObject:info]; // 保存完整的字典，方便 cell 显示 bgmName 和 singerName
        } else {
            NSLog(@"警告：找不到文件 %@", fileName);
        }
    }

    self.musicPaths = musicPaths;
    self.musicInfos = musicInfos;

    NSLog(@"加载完成：共 %lu 首音乐", (unsigned long)self.musicPaths.count);
}

#pragma mark - UITableViewDataSource

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section {
    return self.musicPaths.count;
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath {
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:@"cell" forIndexPath:indexPath];
    NSString *bgmName = [self.musicInfos[indexPath.item] objectForKey:@"bgmName"];
    NSString *singerName = [self.musicInfos[indexPath.item] objectForKey:@"singerName"];
    cell.textLabel.text = [NSString stringWithFormat:@"%@ - %@",bgmName,singerName];

    return cell;
}

#pragma mark - UITableViewDelegate
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath {
    NSString *selectedMusicPath = self.musicPaths[indexPath.row];
    TAVBgmResource *resource = [[TAVBgmResource alloc] initWithPath:selectedMusicPath];
    if(indexPath.item < self.musicInfos.count) {
        resource.bgmName = [self.musicInfos[indexPath.item] objectForKey:@"bgmName"];
        resource.singerName = [self.musicInfos[indexPath.item] objectForKey:@"singerName"];
        resource.extraInfo = [self.musicInfos[indexPath.item] objectForKey:@"extraInfo"];

    }
    // 调用回调
    if (self.musicSelectedCallback) {
        self.musicSelectedCallback(resource);
    }
    [self closeView];
}

@end
