#import "MaterialProvider.h"
#import <TAVMagicUIKit/ITAVMaterialProvider.h>
#import <TAVMagicUIKit/TAVAlbumResource.h>
#import <TAVMagicUIKit/TAVBgmResource.h>
#import "BgmSelectorView.h"
#import <TAVMagicUIKit/TAVAlbumPickerController.h>
#import <TAVMagicUIKit/TAVAlbumPickerFactory.h>
#import <TAVMagicUIKit/TAVAlbumAuthorizer.h>
#import <TAVMagicUIKit/TAVAlbumResource.h>
#import <TAVMagicUIKit/TAVAlbumAssetCustomCollectionViewCell.h>
#import <TAVMagicUIKit/NSObject+TAV.h>

@interface MaterialProvider () <TAVAlbumPickerControllerDelegate>
@property (nonatomic, copy) void (^albumSelectedCallback)(NSArray<TAVAlbumResource *> *resourceList); // 回调
@property (nonatomic, strong) UIProgressView *progressView;
@property (nonatomic, copy) NSString *defaultSandBoxTmpDir;


@end

@implementation MaterialProvider

- (instancetype)init {
    if(self = [super init]) {
        // 创建新的文件夹名称
        NSString *folderName = @"MagicCut";
        _defaultSandBoxTmpDir = [[NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES) firstObject] stringByAppendingPathComponent:folderName];
        // 每次编辑前先清除一下缓存
        [self removeDefaultSandBoxTmpDir];
        // 确保输出目录存在
        [self createDefaultSandBoxTmpDir];
        // 退出时清除
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(removeDefaultSandBoxTmpDir)
        name:UIApplicationWillTerminateNotification object:nil];
    }
    return self;
}

- (void)dealloc {
    self.progressView = nil;
    self.defaultSandBoxTmpDir = nil;
    self.albumSelectedCallback = nil;
}

- (UIActivityIndicatorView *)creatIndicatorViewWithFrame:(CGRect)frame {
    UIActivityIndicatorView *activityView = [[UIActivityIndicatorView alloc] initWithActivityIndicatorStyle:UIActivityIndicatorViewStyleWhiteLarge];
    activityView.frame = frame;
    activityView.hidesWhenStopped = YES;
    return activityView;
}

#pragma mark - ITAVMaterialProvider
- (void)getBgmWithHandler:(handleBgmSelectResultFunc)handler {
    BgmSelectorView *view = [[BgmSelectorView alloc] initWithFrame:[UIScreen mainScreen].bounds];
    //    // 设置回调
    view.musicSelectedCallback = ^(TAVBgmResource * _Nonnull resource) {
        handler(resource);
    };

    [[self tav_findVisibleViewController].view addSubview:view];
}

- (void)getAlbumFilesForType:(TAVAlbumOptions *)option withHandler:(handleAlbumSelectResultFunc)handler {
    self.albumSelectedCallback = handler;
    [TAVAlbumAuthorizer checkAlbumAuthorize:^(PHAuthorizationStatus status) {
        [self pushAlbumPickController:option];
    }];
    
}

- (void)pushAlbumPickController:(TAVAlbumOptions *)option {
    TAVAlbumPickerOptions *options = [TAVAlbumPickerFactory defaultAlbumPickerOptions];
    options.maximumSelectionCount = option.count;
    options.autoDismiss = NO;
    options.customViewOptions.albumAssetCollectionViewCellClass = TAVAlbumAssetCustomCollectionViewCell.class;
//    options.customViewOptions.albumPreviewViewControlleClass = TAVAlbumCustomPreviewViewController.class;
    
    switch (option.type) {
        case TAVResourceVideoType:
            options.mediaType = TAVAlbumAssetTypeVideo;
            break;
        case TAVResourcePhotoType:
            options.mediaType = TAVAlbumAssetTypeImage;
        default:
            break;
    }
    
    TAVAlbumPickerController *pickerController = [[TAVAlbumPickerController alloc] initWithOptions:options];
    pickerController.modalPresentationStyle = UIModalPresentationFullScreen;
    pickerController.pickerDelegate = self;
    
    [[self tav_findVisibleViewController] presentViewController:pickerController animated:YES completion:nil];
}

#pragma mark - TAVAlbumPickerControllerDelegate
- (void)albumPickerController:(TAVAlbumPickerController *)pickerController didFinishPickingAlbumAssets:(NSArray<TAVAlbumAsset*> *)albumAssets
{
    UIActivityIndicatorView *activityView = [self creatIndicatorViewWithFrame:pickerController.view.bounds];
    [pickerController.view addSubview:activityView];
    [activityView startAnimating];
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_HIGH, 0), ^{
        NSMutableArray *resources = [NSMutableArray array];
        for (TAVAlbumAsset *mediaItem in albumAssets) {
            TAVAlbumResource *resource;
            if (mediaItem.mediaType == TAVAlbumAssetTypeVideo) {
                resource = [self videoResourceWithPHAsset:mediaItem.asset];
                if (self.prepareEditor) {
                    [self.prepareEditor addVideoClip:resource.path];
                }
            } else if (mediaItem.mediaType == TAVAlbumAssetTypeImage) {
                // livePhoto
                if (mediaItem.asset.mediaSubtypes & PHAssetMediaSubtypePhotoLive) {
                    resource = [self livePhotoResourceWithPHAsset:mediaItem.asset];
                    if (self.prepareEditor) {
                        [self.prepareEditor addVideoClip:resource.path];
                    }
                } else {
                    resource = [self imageResourceWithPHAsset:mediaItem.asset];
                    if (self.prepareEditor) {
                        [self.prepareEditor addImageClip:resource.path duration:kCMTimeZero];
                    }
                }
            }
            [resources addObject:resource];
        }
        if (self.prepareEditor) {
            [self.prepareEditor preparePlayerItem];
        }
        dispatch_async(dispatch_get_main_queue(), ^{
            [activityView stopAnimating];
            self.albumSelectedCallback(resources);
            self.albumSelectedCallback = nil;
            [pickerController dismissViewControllerAnimated:YES completion:nil];
        });
    });
}

- (TAVAlbumResource *)videoResourceWithPHAsset:(PHAsset *)phAsset {
    __block AVURLAsset *urlAsset;
    dispatch_semaphore_t semaphore = dispatch_semaphore_create(0);
    PHVideoRequestOptions *options = [[PHVideoRequestOptions alloc] init];
    options.networkAccessAllowed = NO;
    [PHImageManager.defaultManager requestAVAssetForVideo:phAsset options:options resultHandler:^(AVAsset * _Nullable asset, AVAudioMix * _Nullable audioMix, NSDictionary * _Nullable info) {
        urlAsset = (AVURLAsset *)asset;
        dispatch_semaphore_signal(semaphore);
    }];
    dispatch_semaphore_wait(semaphore, DISPATCH_TIME_FOREVER);
    NSString *fileNameWithoutExtension = [[urlAsset URL].path lastPathComponent] ;
    NSString *toPath = [self.defaultSandBoxTmpDir stringByAppendingPathComponent:fileNameWithoutExtension];
    [self copyFileFromPath:[urlAsset URL].path toPath:toPath];
    TAVAlbumResource *videoRes = [[TAVAlbumResource alloc] initWithPath:toPath forType:TAVResourceVideoType];
    return videoRes;
}

- (TAVAlbumResource *)imageResourceWithPHAsset:(PHAsset *)phAsset {
    PHAssetResource *imageResource = [PHAssetResource assetResourcesForAsset:phAsset].lastObject;
    NSURL *imageUrl = [imageResource valueForKey:@"privateFileURL"];
    NSString *originalFileName = imageResource.originalFilename;
    NSString *toPath = [self.defaultSandBoxTmpDir stringByAppendingPathComponent:originalFileName];
    [self copyFileFromPath:imageUrl.path toPath:toPath];
    TAVAlbumResource *imageRes = [[TAVAlbumResource alloc] initWithPath:toPath forType:TAVResourcePhotoType];
    return imageRes;
}

- (TAVAlbumResource *)livePhotoResourceWithPHAsset:(PHAsset *)phAsset {
    PHAssetResource *imageResource = [PHAssetResource assetResourcesForAsset:phAsset].lastObject;
    NSURL *videoUrl = [imageResource valueForKey:@"privateFileURL"];
    NSString *originalFileName = imageResource.originalFilename;
    NSString *toPath = [self.defaultSandBoxTmpDir stringByAppendingPathComponent:originalFileName];
    [self copyFileFromPath:videoUrl.path toPath:toPath];
    TAVAlbumResource *imageRes = [[TAVAlbumResource alloc] initWithPath:toPath forType:TAVResourceVideoType];
    return imageRes;
}

#pragma mark - FileManager
- (BOOL)copyFileFromPath:(NSString *)fromPath toPath:(NSString *)toPath {
    if (![self fileExistsAtPath:fromPath]) return NO;
    BOOL remove = YES;
    if ([self fileExistsAtPath:toPath]) {
        remove = [self removeFileAtPath:toPath];
    }
    NSError *error = nil;
    BOOL copy = [[NSFileManager defaultManager] copyItemAtPath:fromPath toPath:toPath error:&error];
    return (remove && copy);
}

- (BOOL)fileExistsAtPath:(NSString *)filePath {
    BOOL isDirectory = NO;
    BOOL result = [[NSFileManager defaultManager] fileExistsAtPath:filePath isDirectory:&isDirectory];
    return result&!isDirectory;
}

- (BOOL)removeFileAtPath:(NSString *)path {
    if (![self fileExistsAtPath:path]) return NO;
    BOOL remove = [[NSFileManager defaultManager] removeItemAtPath:path error:NULL];
    return remove;
}

- (void)createDefaultSandBoxTmpDir {
    // 确保输出目录存在
    NSFileManager *fileManager = [NSFileManager defaultManager];
    if (![fileManager fileExistsAtPath:self.defaultSandBoxTmpDir]) {
        NSError *error = nil;
        [fileManager createDirectoryAtPath:self.defaultSandBoxTmpDir withIntermediateDirectories:YES attributes:nil error:&error];
        if (error) {
            NSLog(@"创建目录失败: %@", error.localizedDescription);
            return; // 处理错误
        }
    }
}

- (void)removeDefaultSandBoxTmpDir {
    if(self.defaultSandBoxTmpDir == nil) {
        return;
    }
    NSFileManager *fileManager = [NSFileManager defaultManager];
    
    // 删除目录及其内容
    if ([fileManager fileExistsAtPath:self.defaultSandBoxTmpDir]) {
        NSError *error = nil;
        NSURL *defaultSandBoxTmpDirURL = [NSURL fileURLWithPath:self.defaultSandBoxTmpDir];
        [fileManager removeItemAtURL:defaultSandBoxTmpDirURL error:&error];
        if (error) {
            NSLog(@"删除目录失败: %@", error.localizedDescription);
        } else {
            NSLog(@"删除目录成功");
        }
    }
}

@end
