//
//  HomeViewController.m
//  TAVMagicDemo
//
//  Created by Robin on 2024/7/22.
//

#import "HomeViewController.h"
#import "TAVTemplateListViewController.h"
#import "AFNetworking/AFNetworking.h"
#import <TAVMagicUIKit/TAVDraftViewController.h>
#import "MaterialProvider.h"
#import "TAVPublishController.h"

#import "HomeInfoView.h"
#import "HomeCreationView.h"
#import "TAVVideoShootViewController.h"
#import <TAVMagicUIKit/UIColor+TAVUtil.h>
#import <TAVMagicUIKit/TAVEditorPreviewController.h>
#import <TAVMagicUIKit/TAVEditorManager.h>
@import Masonry;

@interface HomeViewController ()<TAVDraftViewControllerDelegate,TAVLogListener>

@property (nonatomic, strong) HomeCreationView *creationView;
@property (nonatomic, strong) HomeInfoView *infoView;
@property (nonatomic, strong) CAGradientLayer *gradientLayer;
@end

@implementation HomeViewController


- (void)viewDidLoad {
    [super viewDidLoad];
    [self.navigationController setNavigationBarHidden:YES animated:NO];

    [self setupUI];
    [self addGradientColor];
    [TAVEditor setLogLevel:TAVLogLevelInfo listener:self];
}

- (void)onLog:(TAVLogLevel)loggerLevel logMessage:(NSString * _Nonnull)logMessage {
    NSLog(@"%@",logMessage);
}

- (void)viewDidDisappear:(BOOL)animated {
    [super viewDidDisappear:animated];
    [self pause];
    [self removeEditorEntryView];
}

- (void)viewDidAppear:(BOOL)animated{
    [super viewDidAppear:animated];
    [self resume];
    
    BOOL hasOpened = [self checkHasOpened];
    if (!hasOpened){
        [self showEditorEntryViewAnimal:NO];
    }
}
- (void)addGradientColor{
    CAGradientLayer *gradientLayer = [CAGradientLayer layer];
    
    // 设置渐变颜色
    UIColor *color1 = [UIColor tvc_colorWithHexString:@"#0F101400"];
    UIColor *color2 = [UIColor tvc_colorWithHexString:@"#0F101499"];
    gradientLayer.colors = @[(id)color2.CGColor, (id)color1.CGColor];
    
    // 设置渐变的起始点和结束点，这里表示从顶部到底部的渐变
    gradientLayer.startPoint = CGPointMake(0.5, 0);
    gradientLayer.endPoint = CGPointMake(0.5, 1);
    
    // 设置渐变图层的大小与视图的大小一致
    gradientLayer.frame = CGRectMake(0, 0, self.view.frame.size.width, 123);
    
    // 将渐变图层添加到视图的图层上
    [self.view.layer insertSublayer:gradientLayer atIndex:0];
    self.gradientLayer = gradientLayer;
}


#pragma mark - Page Action
- (void)configPreviewContrller:(TAVEditorPreviewController *)previewVC {
    previewVC.materialProvider = self.materialProvider;
    previewVC.editInterceptor = self.editInterceptor;
    __weak __typeof(TAVEditorPreviewController *)weakPreviewVC = previewVC;

    [previewVC setNextClickCallback:^{
        __strong typeof(TAVEditorPreviewController *) strongPreviewVC = weakPreviewVC;
        TAVPublishController *publishVC = [[TAVPublishController alloc] initWithEditor:strongPreviewVC.editor];
        [self.navigationController pushViewController:publishVC animated:YES];
    }];
    [self.navigationController setNavigationBarHidden:YES];
}

- (void)gotoPreviewControllerWithResource:(NSArray<TAVAlbumResource *> *)resources {
//    TAVEditorPreviewController *previewVC = [[TAVEditorPreviewController alloc] initWithAssetPath:nil resources:resources];
    TAVEditorPreviewController *previewVC = [[TAVEditorPreviewController alloc] initWithTavEditor:self.materialProvider.prepareEditor];
    self.materialProvider.prepareEditor = nil;
    [self configPreviewContrller:previewVC];
    [self.navigationController pushViewController:previewVC animated:YES];
}

- (void)pushPreviewControllerWithDraft:(TAVDraft *)draft {
    TAVEditorPreviewController *previewVC = [[TAVEditorPreviewController alloc] initWithAssetPath:nil draft:draft];
    [self configPreviewContrller:previewVC];
    [self.navigationController pushViewController:previewVC animated:YES];
}

#pragma mark - Short Video
- (void)scrollToVideoIndex:(NSInteger )videoIndex
                videoModel:(TUIPlayerDataModel *)videoModel {
    NSDictionary *extInfo = videoModel.extInfo;
    self.infoView.titleLabel.text = [extInfo objectForKey:@"name"];
    self.infoView.subTitleLabel.text = [extInfo objectForKey:@"title"];
    [self.infoView setNeedsLayout];
}

#pragma mark - UIAction
- (void)setupUI{
    self.infoView = [[HomeInfoView alloc] initWithFrame:CGRectZero];
    [self.view addSubview:self.infoView];
//    self.infoView.titleLabel.text = @"@测试测试";
//    self.infoView.subTitleLabel.text = @"描述文案描述文";
    [self.infoView.creationButton setTitle:NSLocalizedString(@"Home.Create", "创作") forState:UIControlStateNormal];
    [self.infoView.creationButton addTarget:self action:@selector(creationAction:) forControlEvents:UIControlEventTouchUpInside];
    [self.infoView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.right.offset(0);
        make.height.mas_equalTo(144);
        make.top.equalTo(self.view.mas_safeAreaLayoutGuideBottom).offset(-110);
    }];
}

- (void)showEditorEntryViewAnimal:(BOOL)animation{
    self.infoView.creationButton.enabled = NO;
    
    if (!self.creationView){
        CGRect frame = CGRectMake(0, self.view.frame.size.height, self.view.bounds.size.width, 272);
        HomeCreationView *creationView = [[HomeCreationView alloc] initWithFrame:frame];
        creationView.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleTopMargin;
        self.creationView = creationView;
        
        [creationView.closeButton addTarget:self action:@selector(creationViewCloseAction:) forControlEvents:UIControlEventTouchUpInside];
        [creationView.videoShootButton addTarget:self action:@selector(videoShootAction:) forControlEvents:UIControlEventTouchUpInside];
        [creationView.videoEditButton addTarget:self action:@selector(shortVideoMakeAction:) forControlEvents:UIControlEventTouchUpInside];
        [creationView.makeSameButton addTarget:self action:@selector(makeSameAction:) forControlEvents:UIControlEventTouchUpInside];
        [creationView.draftBoxButton addTarget:self action:@selector(draftAction:) forControlEvents:UIControlEventTouchUpInside];
    }
    
    UIEdgeInsets safeAreaInsets = self.view.safeAreaInsets;
    CGRect newFrame = CGRectMake(0, self.view.bounds.size.height - 238 - safeAreaInsets.bottom, self.view.bounds.size.width, 272);
    [self.view addSubview:self.creationView];
    if (animation){
        [UIView animateWithDuration:0.15 delay:0 options:UIViewAnimationOptionCurveEaseInOut animations:^{
            self.creationView.frame = newFrame;
        } completion:nil];
    } else {
        self.creationView.frame = newFrame;
    }
}

- (void)removeEditorEntryView{
    self.infoView.creationButton.enabled = YES; 
    
    if (!self.creationView){
        return;
    }
    
    CGRect newFrame = CGRectMake(0, self.view.frame.size.height, self.view.bounds.size.width, 272);
    [UIView animateWithDuration:0.15 animations:^{
        self.creationView.frame = newFrame;
    } completion:^(BOOL finished) {
        [self.creationView removeFromSuperview];
    }];
}

#pragma mark - Button Action
- (void)creationAction:(UIButton *)sender {
    [self pause];
    [self showEditorEntryViewAnimal:YES];
}

- (void)creationViewCloseAction:(UIButton *)sender {
    [self resume];
    [self removeEditorEntryView];
}

- (void)videoShootAction:(UIButton *)sender {
    TAVVideoShootViewController *videoShootVC = [[TAVVideoShootViewController alloc] init];
    [self.navigationController pushViewController:videoShootVC animated:YES];
}

- (void)makeSameAction:(UIButton *)sender {
    [self pause];
    [self removeEditorEntryView];
    NSString *path = [[NSBundle mainBundle] pathForResource:@"template" ofType:@"bundle"];
    TAVTemplateListViewController *templateVC = [TAVTemplateListViewController templateViewController:path];
    if (templateVC) {
        [self.navigationController pushViewController:templateVC animated:YES];
    }
}

- (void)shortVideoMakeAction:(UIButton *)sender {
    [self pause];
    [self removeEditorEntryView];
    NSString *bundlePath = [[NSBundle mainBundle] pathForResource:@"LightCore" ofType:@"bundle"];
    self.materialProvider.prepareEditor = [TAVEditor editorWithRenderSize:CGSizeZero assetsPath:bundlePath];
    __weak __typeof(self)weakSelf = self;
    [self.materialProvider getAlbumFilesForType:[TAVAlbumOptions new] withHandler:^(NSArray<TAVAlbumResource *> *resourceList) {
        [weakSelf gotoPreviewControllerWithResource:resourceList];
    }];
}

- (void)draftAction:(UIButton *)sender {
    [self pause];
    [self removeEditorEntryView];
    TAVDraftViewController *viewController = [[TAVDraftViewController alloc] initWithAssetPath:nil];
    viewController.materialProvider = self.materialProvider;
    viewController.delegate = self;
    [self.navigationController setNavigationBarHidden:YES];
    [self.navigationController pushViewController:viewController animated:YES];
}

#pragma mark TAVDraftViewControllerDelegate
- (void)draftViewItemClick:(TAVDraft *)draft {
    [self pushPreviewControllerWithDraft:draft];
}

#pragma mark - private
const static NSString *KAPP_FIRST_OPEN_KEY = @"kAppFirstOpenKey";
- (BOOL)checkHasOpened{
    BOOL hasOpened = [[NSUserDefaults standardUserDefaults] boolForKey:(NSString *)KAPP_FIRST_OPEN_KEY];
    [[NSUserDefaults standardUserDefaults] setBool:YES forKey:(NSString *)KAPP_FIRST_OPEN_KEY];
    return hasOpened;
}

#pragma mark - Getter
- (MaterialProvider *)materialProvider {
    if (!_materialProvider) {
        _materialProvider = [[MaterialProvider alloc] init];
    }
    return _materialProvider;
}

- (EditInterceptor *)editInterceptor {
    if (!_editInterceptor) {
        _editInterceptor = [[EditInterceptor alloc] init];
    }
    return _editInterceptor;
}

@end
