//
//  TAVTemplateDetailViewController.m
//  TAVMagicDemo
//
//  Created by Robin on 2024/7/23.
//

#import "TAVTemplateDetailViewController.h"
#import "TAVTemplateDetailViewController+CutSame.h"
#import "TAVTemplateNavView.h"
#import "TAVTemplateSet.h"
#import <TAVMagicUIKit/UIColor+TAVUtil.h>
@import Masonry;
@import AVKit;

@interface TAVTemplateDetailViewController ()
@property (nonatomic, strong) TAVTemplateNavView *navView;
@property (nonatomic, strong) AVPlayerLayer *playerLayer;
@property (nonatomic, strong) AVPlayer *player;
@property (nonatomic, strong) UIImageView *imageView;
@property (nonatomic, strong) UILabel *timeLabel;
@property (nonatomic, strong) UILabel *titleLabel;
@property (nonatomic, strong) UILabel *desLabel;
@property (nonatomic, strong) UIButton *cutSameBtn;
@property (nonatomic, strong) UIButton *playBtn;
@end

@implementation TAVTemplateDetailViewController

- (instancetype)init:(TAVTemplateItem *)item {
    self = [super init];
    if (self) {
        _item = item;
    }
    return self;
}

- (void)viewDidLoad {
    [super viewDidLoad];
    self.view.backgroundColor = UIColor.blackColor;
    
    [self setupSeniorEditor:nil];
    
    BOOL isVideo = [_item.video.pathExtension isEqualToString:@"mp4"];
    if (isVideo) {
        [self.view.layer addSublayer:self.playerLayer];
        [self.view addSubview:self.playBtn];
        [self.playBtn mas_makeConstraints:^(MASConstraintMaker *make) {
            make.centerX.mas_equalTo(self.view.mas_centerX);
            make.centerY.mas_equalTo(self.view.mas_centerY);
            make.width.height.mas_equalTo(100);
        }];
    } else {
        [self.view addSubview:self.imageView];
    }
    
    [self.view addSubview:self.navView];
//    [self.view addSubview:self.timeLabel];
    [self.view addSubview:self.titleLabel];
    [self.view addSubview:self.desLabel];
    [self.view addSubview:self.cutSameBtn];

    [self addNotification];
    
    UITapGestureRecognizer *tap = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(tapAction)];
    [self.view addGestureRecognizer:tap];

    [self.navView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.right.offset(0);
        make.top.equalTo(self.view.mas_safeAreaLayoutGuideTop);
        make.height.offset(44);
    }];
//    [self.timeLabel mas_makeConstraints:^(MASConstraintMaker *make) {
//        make.left.offset(20);
//        make.bottom.equalTo(self.titleLabel.mas_top).offset(-10);
//    }];
    [self.titleLabel mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.offset(20);
        make.bottom.equalTo(self.desLabel.mas_top).offset(-10);
    }];
    [self.desLabel mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.offset(20);
        make.bottom.equalTo(self.view.mas_safeAreaLayoutGuideBottom).offset(-10);
    }];
    [self.cutSameBtn mas_makeConstraints:^(MASConstraintMaker *make) {
        make.right.offset(-20);
        make.bottom.equalTo(self.view.mas_safeAreaLayoutGuideBottom).offset(-10);
    }];
}

- (void)addKvoObserver
{
    [self.player addObserver:self
                  forKeyPath:@"rate"
                     options:NSKeyValueObservingOptionNew
                     context:nil];
}

- (void)removeKvoObserver
{
    [self.player removeObserver:self forKeyPath:@"rate"];
}

- (void)observeValueForKeyPath:(NSString *)keyPath
                      ofObject:(id)object
                        change:(NSDictionary<NSKeyValueChangeKey,id> *)change
                       context:(void *)context {
    if (object == self.player && [keyPath isEqualToString:@"rate"]) {
        float rate = [change[NSKeyValueChangeNewKey] floatValue];
        if (rate == 0) {
            self.playBtn.hidden = NO;
        } else if (rate == 1) {
            self.playBtn.hidden = YES;
        }
    }
}

- (void)addNotification
{
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(appWillEnterForeground:) name:UIApplicationWillEnterForegroundNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(appDidEnterBackground:) name:UIApplicationDidEnterBackgroundNotification object:nil];
}

- (void)removeNotification
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

- (void)appWillEnterForeground:(NSNotification *)note
{
    [self.player play];
}

- (void)appDidEnterBackground:(NSNotification *)note
{
    [self.player pause];
}

- (void)tapAction
{
    [self.player pause];
}

- (void)playClick
{
    [self.player play];
}

- (void)viewWillAppear:(BOOL)animated {
    [super viewWillAppear:animated];
    [self.player play];
    [self addNotification];
}

- (void)viewDidDisappear:(BOOL)animated {
    [super viewDidDisappear:animated];
    [self.player pause];
    [self removeNotification];
}

- (void)viewWillLayoutSubviews{
    [super viewWillLayoutSubviews];
    CGRect frame = self.view.frame;
    frame.origin.y += self.view.safeAreaInsets.top;
    frame.size.height -= (70 + self.view.safeAreaInsets.top + self.view.safeAreaInsets.bottom);
    if (_playerLayer){
        _playerLayer.frame = frame;
        self.playerLayer.mask.frame = self.playerLayer.bounds;
        self.playerLayer.mask = self.playerLayer.mask;
    } else if (_imageView){
        _imageView.frame = frame;
    }
}

- (void)dealloc
{
    [self removeNotification];
    [self removeKvoObserver];
}

#pragma mark - lazy
- (TAVTemplateNavView *)navView {
    if (!_navView) {
        _navView = [[TAVTemplateNavView alloc] initWithTitle:@"" sender:self];
    }
    return _navView;
}

- (AVPlayerLayer *)playerLayer {
    if (!_playerLayer) {
        NSURL *url = [NSURL fileURLWithPath:_item.video];
        AVPlayerItem *item = [AVPlayerItem playerItemWithURL:url];
        AVPlayer *player = [AVPlayer playerWithPlayerItem:item];
        
        __weak typeof(player) wsPlayer = player;
        [player addBoundaryTimeObserverForTimes:@[[NSValue valueWithCMTime:item.asset.duration]] queue:dispatch_get_main_queue() usingBlock:^{
            [wsPlayer seekToTime:CMTimeMake(0, NSEC_PER_SEC)];
            dispatch_async(dispatch_get_main_queue(), ^{
                [wsPlayer play];
            });
        }];
        
        _playerLayer = [AVPlayerLayer playerLayerWithPlayer:player];
        _playerLayer.videoGravity = AVLayerVideoGravityResizeAspect;
        CGRect frame = self.view.bounds;
        frame.size.height -= (74 + self.view.safeAreaInsets.bottom);
        _playerLayer.frame = frame;
        CALayer *maskLayer = [CAShapeLayer layer];
        maskLayer.frame = _playerLayer.bounds;
        maskLayer.cornerRadius = 12;
        maskLayer.backgroundColor = UIColor.redColor.CGColor;
        _playerLayer.mask = maskLayer;
        
        _player = player;
        [player play];
        [self addKvoObserver];
    }
    return _playerLayer;
}

- (UIButton *)playBtn
{
    if (!_playBtn) {
        _playBtn = [UIButton buttonWithType:UIButtonTypeCustom];
        [_playBtn setImage:[UIImage imageNamed:@"feed_icon_display"] forState:UIControlStateNormal];
        _playBtn.frame = CGRectMake(0, 0, 100, 100);
        [_playBtn addTarget:self action:@selector(playClick) forControlEvents:UIControlEventTouchUpInside];
    }
    return _playBtn;
}

- (UIImageView *)imageView {
    if (!_imageView) {
        UIImage *image = [UIImage imageWithContentsOfFile:_item.video];
        _imageView = [[UIImageView alloc] initWithImage:image];
        _imageView.frame = self.view.bounds;
        _imageView.layer.cornerRadius = 12;
        _imageView.clipsToBounds = YES;
    }
    return _imageView;
}

- (UILabel *)timeLabel {
    if (!_timeLabel) {
        _timeLabel = UILabel.new;
        _timeLabel.text = _item.durationDesc;
        _timeLabel.textColor = UIColor.whiteColor;
        _timeLabel.font = [UIFont systemFontOfSize:14];
    }
    return _timeLabel;
}

- (UILabel *)titleLabel {
    if (!_titleLabel) {
        _titleLabel = UILabel.new;
        _titleLabel.text = _item.name;
        _titleLabel.textColor = [UIColor tvc_colorWithHexString:@"#ffffffe6"];
        _titleLabel.font = [UIFont boldSystemFontOfSize:16];
    }
    return _titleLabel;
}

- (UILabel *)desLabel {
    if (!_desLabel) {
        _desLabel = UILabel.new;
        _desLabel.text = _item.replaceDesc;
        _desLabel.textColor = [UIColor tvc_colorWithHexString:@"#ffffff8c"];
        _desLabel.font = [UIFont systemFontOfSize:12];
    }
    return _desLabel;
}

- (UIButton *)cutSameBtn {
    if (!_cutSameBtn) {
        _cutSameBtn = [UIButton buttonWithType:UIButtonTypeSystem];
        [_cutSameBtn setTitle:NSLocalizedString(@"Home.Creation.Template", "剪同款") forState:UIControlStateNormal];
        [_cutSameBtn setTitleColor:UIColor.whiteColor forState:UIControlStateNormal];
        [_cutSameBtn setContentEdgeInsets:UIEdgeInsetsMake(10, 20, 10, 20)];
        [_cutSameBtn setBackgroundColor:[UIColor tvc_colorWithHexString:@"#FF6500"]];
        [_cutSameBtn sizeToFit];
        _cutSameBtn.layer.cornerRadius = _cutSameBtn.frame.size.height / 2;
        _cutSameBtn.layer.masksToBounds = YES;
        [_cutSameBtn addTarget:self action:@selector(cutSameAction:) forControlEvents:UIControlEventTouchUpInside];
    }
    return _cutSameBtn;
}

@end
