//
//  TAVTemplateEditorViewController.m
//  TAVMagicApp
//
//  Created by v_zwtzzhou on 2024/7/23.
//

#import "TAVTemplateEditorViewController.h"
#import "TAVVideoPlayerViewController.h"
#import <TAVMagicUIKit/TAVTextToastView.h>
#import <TAVMagicUIKit/UIColor+TAVUtil.h>
#import "TAVTemplateVideoSetView.h"
@interface TAVTemplateEditorViewController ()
@property (nonatomic, strong) UIButton *backButton;
@property (nonatomic, strong) UIButton *nextButton;
@property (nonatomic, strong) UIButton *resolutionButton;
@property (nonatomic, strong) TAVEditorGenerateConfig *generateConfig;
@property (nonatomic, strong) TAVVideoPlayerViewController *previewController;
@property (nonatomic, strong) TAVTemplateVideoSetView *videoSetView;
@property (nonatomic, assign) BOOL showResolutionView;

@end

@implementation TAVTemplateEditorViewController

- (void)viewDidLoad{
    [super viewDidLoad];
    
    [self setupPlayer];
    [self setupUI];
}

- (void)viewDidLayoutSubviews{
    [super viewDidLayoutSubviews];
    
    CGSize fsize = self.view.bounds.size;
    self.backButton.frame = CGRectMake(6, 6 + self.view.safeAreaInsets.top, 32, 32);
    self.nextButton.frame = CGRectMake(fsize.width - 96, CGRectGetMidY(self.backButton.frame) - 16, 80, 32);
    self.resolutionButton.frame = CGRectMake(CGRectGetMinX(self.nextButton.frame) - 81, CGRectGetMinY(self.nextButton.frame), 73, 32);
    self.previewController.view.frame = CGRectMake(0, CGRectGetMaxY(self.nextButton.frame) + 7, fsize.width, fsize.height - CGRectGetMaxY(self.nextButton.frame) - 7);
    
}

- (void)viewSafeAreaInsetsDidChange{
    [super viewSafeAreaInsetsDidChange];
    [self.view setNeedsUpdateConstraints];
}

- (void)setupUI{
    self.view.backgroundColor = UIColor.blackColor;
    
    [self.view addSubview:self.backButton];
    [self.view addSubview:self.nextButton];
    [self.view addSubview:self.resolutionButton];
}

- (void)setupPlayer{
    [self addChildViewController:self.previewController];
    [self.view addSubview:self.previewController.view];
    [self.view setNeedsUpdateConstraints];
    
    self.previewController.editor = self.editor;
    [self.previewController setBgmPath:self.bgmPath];
    [self.editor setPreview:self.previewController.previewParam];
    [self.previewController play];
}

- (void)setSelectedAsset:(NSArray<TAVAlbumAsset *> *)selectedAsset {
    _selectedAsset = selectedAsset;
    self.previewController.assets = selectedAsset;
}

#pragma mark - Save
- (void)generateVideo
{
    NSString *tempDir = NSTemporaryDirectory();
    NSString *appLib = [tempDir stringByAppendingString:@"/Caches"];
    NSString *resDir = [appLib stringByAppendingPathComponent:@"tavXmagic"];

    // 暂停视频播放
    [self.previewController pause];
    [TAVTextToastView showToast:[NSString stringWithFormat:@"视频生成中"] duration:1];
    [self.editor quickGenerateVideo:self.generateConfig outputPath:resDir progress:^(CGFloat progress) {
        NSLog(@"progress : %f",progress);
        [TAVTextToastView showToast:[NSString stringWithFormat:@"视频生成中:%.0f%%",progress * 100] duration:0.01];
    } completion:^(NSError * _Nonnull error, NSString * _Nonnull outputPath) {
        if (error) {
            NSLog(@"generateVideo error : %@",error);
            return;
        }
        BOOL videoCompatible = UIVideoAtPathIsCompatibleWithSavedPhotosAlbum(outputPath);
        if (videoCompatible) {
            UISaveVideoAtPathToSavedPhotosAlbum(outputPath, self, @selector(video:didFinishSavingWithError:contextInfo:), nil);
        } else {
            NSLog(@"该视频无法保存至相册");
        }
        
    }];
    
}

- (void)video:(NSString *)videoPath didFinishSavingWithError:(NSError *)error contextInfo:(void *)contextInfo {
    // 保存相册后删除缓存
    [[NSFileManager defaultManager] removeItemAtPath:videoPath error:nil];
    if (error) {
        NSLog(@"保存视频到相册失败：%@", videoPath);
        NSString *toastText = NSLocalizedString(@"Failed.to.save.video","保存视频失败");
        [TAVTextToastView showToast:toastText duration:2];
    } else {
        NSLog(@"保存视频到相册成功");
        NSString *toastText = NSLocalizedString(@"Video.saved.successfully","保存视频到相册成功");
        [TAVTextToastView showToast:toastText duration:2];
    }
}

#pragma mark - actions
- (void)backAction:(UIButton *)sender {
    [self.navigationController popViewControllerAnimated:YES];
}

- (void)nextAction:(UIButton *)sender {
    [self generateVideo];
}

- (void)resolutionAction:(UIButton *)sender {
    self.showResolutionView = !self.showResolutionView;
    [UIView animateWithDuration:0.3 animations:^{
        CGAffineTransform transform = self.showResolutionView ? CGAffineTransformMakeRotation(M_PI) : CGAffineTransformIdentity;
        self.resolutionButton.imageView.transform = transform;
    } completion:^(BOOL finished) {
        [self.resolutionButton setTitle:self.videoSetView.resolutionString forState:UIControlStateNormal];
        [self updateResolutionBtnInsets];
    }];
    if (self.showResolutionView) {
        self.videoSetView.type = TAVVideoSetTypeVideoInfo;
        [self.view addSubview:self.videoSetView];
        [self.videoSetView showActionSheet];
    } else {
        [self.videoSetView hideActionSheet];
    }
    
}

- (void)finishBtnClick:(UIButton *)sender {
    self.generateConfig = [[TAVEditorGenerateConfig alloc] init];
    self.generateConfig.videoConfig.resolution = self.videoSetView.resolutionSize;
    self.generateConfig.videoConfig.frameRate = self.videoSetView.frameRate;
    self.generateConfig.videoConfig.videoBitRate = self.videoSetView.videoBitRate;
    [self resolutionAction:nil];
}

- (void)updateResolutionBtnInsets {
    CGFloat spacing = 5.0;
    CGSize imageSize = self.resolutionButton.imageView.intrinsicContentSize;
    CGSize titleSize = self.resolutionButton.titleLabel.intrinsicContentSize;
    self.resolutionButton.titleEdgeInsets = UIEdgeInsetsMake(0, -imageSize.width, 0, imageSize.width + spacing);
    self.resolutionButton.imageEdgeInsets = UIEdgeInsetsMake(0, titleSize.width + spacing, 0, -titleSize.width);
}

#pragma mark - daily
- (TAVVideoPlayerViewController *)previewController{
    if (!_previewController) {
        _previewController = [[TAVVideoPlayerViewController alloc] init];
    }
    return _previewController;
}

- (TAVTemplateVideoSetView *)videoSetView {
    if (!_videoSetView) {
        _videoSetView = [[TAVTemplateVideoSetView alloc] initWithFrame:CGRectMake(0, self.view.bounds.size.height - 284, self.view.bounds.size.width, 284)];
        [_videoSetView.rightButton addTarget:self action:@selector(finishBtnClick:) forControlEvents:UIControlEventTouchUpInside];
    }
    return _videoSetView;
}

- (UIButton *)backButton{
    if (!_backButton){
        _backButton = [UIButton buttonWithType:UIButtonTypeCustom];
        [_backButton setImage:[UIImage imageNamed:@"icon_edit_nav_back_white"] forState:UIControlStateNormal];
        [_backButton addTarget:self action:@selector(backAction:) forControlEvents:UIControlEventTouchUpInside];
    }
    return _backButton;
}

- (UIButton *)nextButton{
    if (!_nextButton){
        _nextButton = [UIButton buttonWithType:UIButtonTypeCustom];
        _nextButton.backgroundColor = [UIColor tvc_colorWithHexString:@"#FF6500"];
        _nextButton.layer.cornerRadius = 16;
        _nextButton.titleLabel.font = [UIFont systemFontOfSize:14];
        [_nextButton setTitle:NSLocalizedString(@"Export","导出")
                      forState:UIControlStateNormal];
        _nextButton.titleLabel.textColor = UIColor.whiteColor;
        [_nextButton addTarget:self action:@selector(nextAction:) forControlEvents:UIControlEventTouchUpInside];
    }
    return _nextButton;
}

- (UIButton *)resolutionButton {
    if (!_resolutionButton) {
        _resolutionButton =[UIButton buttonWithType:UIButtonTypeCustom];
        _resolutionButton.backgroundColor = [UIColor tvc_colorWithHexString:@"#2B2C30"];
        _resolutionButton.layer.cornerRadius = 16;
        _resolutionButton.titleLabel.font = [UIFont systemFontOfSize:14];
        [_resolutionButton setTitle:@"1080P" forState:UIControlStateNormal];
        _resolutionButton.titleLabel.textColor = [UIColor whiteColor];
        [_resolutionButton setImage:[UIImage imageNamed:@"icon_selector_nav_arrow"] forState:UIControlStateNormal];
        [self updateResolutionBtnInsets];
        [_resolutionButton addTarget:self action:@selector(resolutionAction:) forControlEvents:UIControlEventTouchUpInside];
        _resolutionButton.hidden = YES; //剪同款不支持设置分辨率，把入口隐藏掉
    }
    return _resolutionButton;
}

@end
