//
//  TAVVideoPlayerViewController.m
//  TAVMagicApp
//
//  Created by v_zwtzzhou on 2024/7/23.
//

#import "TAVVideoPlayerViewController.h"
#import <CoreMedia/CoreMedia.h>
#import <TAVMagicUIKit/UIColor+TAVUtil.h>
#import "TAVTemplateEditorBottomView.h"
#import "MaterialProvider.h"

@interface TAVVideoPlayerViewController ()<TAVTemplateEditorBottomViewDelegate>
@property (nonatomic, readwrite) TAVPreviewParam *previewParam;
@property (nonatomic, readwrite) UIImageView *thumbImageView;
@property (nonatomic, readwrite) UIView *playerView;
@property (nonatomic, strong) TAVTemplateEditorBottomView *bottomView;
@property (nonatomic, strong) UIButton *bgmBtn;
@property (nonatomic, strong) UIButton *playButton;
@property (nonatomic, strong) MaterialProvider *materialProvider;

@property (nonatomic, assign) BOOL shouldPlay;
@end

@implementation TAVVideoPlayerViewController

- (void)viewDidLoad{
    [super viewDidLoad];
//    self.view.backgroundColor = UIColor.redColor;
    
    self.shouldPlay = NO;
    [self setupUI];
    [self setupPreviewParam];
}

- (CGFloat)viewBottomSafeHeight{
    return self.view.safeAreaInsets.bottom;
}


- (void)viewDidLayoutSubviews{
    [super viewDidLayoutSubviews];

    CGFloat bottomHeight = 160 + self.view.safeAreaInsets.bottom;
    CGSize fsize = self.view.frame.size;
    self.playButton.bounds = CGRectMake(0, 0, 100, 100);
    self.playButton.center = CGPointMake(fsize.width / 2, (fsize.height - bottomHeight) / 2);
    self.playerView.frame = CGRectMake(0, 0, fsize.width, fsize.height - bottomHeight);
    self.bottomView.frame = CGRectMake(0, fsize.height - bottomHeight, fsize.width, bottomHeight);
    self.bgmBtn.frame = CGRectMake((fsize.width - 180) / 2, 5, 180, 34);
    self.bgmBtn.layer.cornerRadius = 17;
}

- (void)setupPreviewParam{
    TAVPreviewParam *previewParam = [[TAVPreviewParam alloc] init];
    previewParam.videoView = self.playerView;
    previewParam.fillModel = TAVPlayerFillModeAspectFit;
    previewParam.observer = self;
    previewParam.loop = YES;
    self.previewParam = previewParam;
}

- (void)updateRenderSize {
}

- (void)setupUI{
    [self.view addSubview:self.playerView];
    [self.view addSubview:self.playButton];
    [self.view addSubview:self.bottomView];
    [self.view addSubview:self.bgmBtn];
}

- (void)setAssets:(NSArray<TAVAlbumAsset *> *)assets {
    _assets = assets;
    self.bottomView.assets = assets;
}

#pragma mark - Public
- (void)play {
    if(!self.editor.isPlaying){
        [self.editor play];
        self.playButton.hidden = YES;
        self.bottomView.playButton.selected = NO;
    }
}

- (void)replay{
    if (![self.editor isPlaying]){
        [self.editor playAtTime:kCMTimeZero];
        self.playButton.hidden = YES;
        self.bottomView.playButton.selected = NO;
    }
}

- (void)pause {
    if (self.editor.isPlaying){
        [self.editor pause];
        self.playButton.hidden = NO;
        self.bottomView.playButton.selected = YES;
    }
}

- (void)resumeFromLastPausing {
    
}

- (void)seekTo:(CMTime)time {
    [self.editor seekToTime:time];
}

- (void)setBgmPath:(NSString *)bgmPath {
    NSArray *pathComponents = [[bgmPath stringByDeletingLastPathComponent] pathComponents];
    if (pathComponents.count) {
        [self.bgmBtn setTitle:pathComponents.lastObject forState:UIControlStateNormal];
    }
}

- (BOOL)isPlaying{
    return self.editor.isPlaying;
}

- (void)setProgress:(CMTime)progress {
    _progress = progress;
    NSTimeInterval progressSecond = CMTimeGetSeconds(progress);
    self.bottomView.progressSlider.value = progressSecond;
    [self refreshTime];
}

- (void)setDuration:(CMTime)duration {
    _duration = duration;
    NSTimeInterval durationSecond = CMTimeGetSeconds(duration);
    self.bottomView.progressSlider.maximumValue = durationSecond;
    [self refreshTime];
}

- (void)refreshTime{
    NSTimeInterval progressSecond = CMTimeGetSeconds(_progress);
    NSTimeInterval durationSecond = CMTimeGetSeconds(_duration);
    self.bottomView.timeLabel.text = [NSString stringWithFormat:@"%@/%@", [self playbackFormatter:progressSecond], [self playbackFormatter:durationSecond]];
}

- (void)didAddBgmResource:(TAVBgmResource *)resource {
    if(!resource || !resource.path) {
        NSLog(@"bgm resource path cant be nil");
        return;
    }
    NSURL *musicURL = [NSURL fileURLWithPath:resource.path];
    AVAsset *asset = [AVURLAsset URLAssetWithURL:musicURL options:nil];
    NSTimeInterval durationSeconds = CMTimeGetSeconds(asset.duration);
    if (CMTIME_COMPARE_INLINE(asset.duration, ==, kCMTimeZero) ||
        CMTIME_IS_INDEFINITE(asset.duration) ||
        isnan(durationSeconds) ||
        durationSeconds < 0) {
        NSLog(@"Duration is invalid.");
        return;
    }
    // add的话 起点是一定是00:00 ，末尾不超过主轨道末尾
    durationSeconds = MIN(self.editor.duration, CMTimeGetSeconds(asset.duration));
    int sourceId = [self.editor addBgm:resource.path startAtVideoTime:kCMTimeZero timeRange:CMTimeRangeMake(kCMTimeZero, CMTimeMakeWithSeconds(durationSeconds, 600))];
    NSMutableDictionary *extraInfo = [NSMutableDictionary new];
    if(resource.singerName) {
        [extraInfo setObject:resource.singerName forKey:@"singerName"];
    }
    if(resource.bgmName) {
        [extraInfo setObject:resource.bgmName forKey:@"bgmName"];
    }
    if(resource.extraInfo) {
        [extraInfo addEntriesFromDictionary:resource.extraInfo];
    }
    [self.editor setBgm:sourceId extraInfo:extraInfo];
    [self.editor flushImmediately];
    //更新btn title
    [self.bgmBtn setTitle:[NSString stringWithFormat:@"%@ - %@",resource.singerName,resource.bgmName] forState:UIControlStateNormal];
    self.bgmBtn.imageEdgeInsets = UIEdgeInsetsMake(0, 10, 0, 15);
    self.bgmBtn.titleEdgeInsets = UIEdgeInsetsMake(0, 15, 0, 10);
}

#pragma mark - Helper
- (NSString *)playbackFormatter:(NSTimeInterval)totalTime {
    int time = (int)floor(totalTime);
    int munites = time/60;
    int second = time%60;
    NSString *formatTime = [NSString stringWithFormat:@"%02d:%02d", munites, second];
    return formatTime;
}

- (BOOL)shouldReplay {
    return ((self.editor.duration - self.editor.currentTime) < DBL_MIN);
}

#pragma mark - TAVEditorPlayerObserver

- (void)mediaPlayerItemDidChange:(AVPlayerItem *)playerItem
{
    if (!playerItem) {
        //移除item的时候不刷新
        return;
    }
    [self updateRenderSize];
    self.progress = kCMTimeZero;
    self.duration = playerItem.duration;

    if ([self.observer respondsToSelector:@selector(mediaPlayerItemDidChange:)]) {
        [self.observer mediaPlayerItemDidChange:playerItem];
    }
}
- (void)mediaPlayerProgressDidChange:(CMTime)currentTime
{
    self.progress = currentTime;
    if ([self.observer respondsToSelector:@selector(mediaPlayerProgressDidChange:)]) {
        [self.observer mediaPlayerProgressDidChange:currentTime];
    }
}

- (void)mediaPlayerStateDidChange:(TAVPlaybackState)state
{
    if ([self.observer respondsToSelector:@selector(mediaPlayerStateDidChange:)]) {
        [self.observer mediaPlayerStateDidChange:state];
    }
}

- (void)mediaPlayerSpeedDidChange:(CGFloat)speed
{
    if ([self.observer respondsToSelector:@selector(mediaPlayerSpeedDidChange:)]) {
        [self.observer mediaPlayerSpeedDidChange:speed];
    }
}

- (void)mediaPlayerDidPlayToEnd
{
    if ([self.observer respondsToSelector:@selector(mediaPlayerDidPlayToEnd)]) {
        [self.observer mediaPlayerDidPlayToEnd];
    }else {
        [self replay];
    }
}

- (void)mediaPlayerVideoRectDidChange:(CGRect)videoRect{
    if ([self.observer respondsToSelector:@selector(mediaPlayerVideoRectDidChange:)]) {
        [self.observer mediaPlayerVideoRectDidChange:videoRect];
    }
}

#pragma mark - Actions
- (void)playAction {
    if([self isPlaying]){
        self.playButton.selected = YES;
        [self pause];
    } else {
        self.playButton.selected = NO;
        if ([self shouldReplay]){
            [self replay];
        } else {
            [self play];
        }
    }
}
    
- (void)didStartDragging {
    self.shouldPlay = [self isPlaying];
    [self pause];
}

- (void)didChangeTime:(float)value {
    self.progress = CMTimeMakeWithSeconds(value, NSEC_PER_SEC);
    [self.editor seekToTime:self.progress];
}

- (void)didEndDragging {
    if (self.shouldPlay){
        [self play];
    }
}

- (void)bgmBtnClick:(UIButton *)sender {
    if (self.materialProvider == nil) {
        self.materialProvider = [[MaterialProvider alloc] init];
    }
    [self.materialProvider getBgmWithHandler:^(TAVBgmResource *bgmResouce) {
        if(!bgmResouce) {
            return;
        }
        [self didAddBgmResource:bgmResouce];
    }];
}


#pragma mark - daily
- (TAVPreviewParam *)previewParam{
    if (!_previewParam){
        TAVPreviewParam *previewParam = [[TAVPreviewParam alloc] init];
        previewParam.videoView = self.playerView;
        previewParam.fillModel = TAVPlayerFillModeAspectFit;
        previewParam.observer = self;
        previewParam.loop = YES;
        _previewParam = previewParam;
    }
    return _previewParam;
}

- (UIImageView *)thumbImageView{
    if (!_thumbImageView) {
        _thumbImageView = [[UIImageView alloc] init];
        // 首帧图的填充模式，应该与素材里的画布填充模式对齐，默认都是Fit
        _thumbImageView.contentMode = UIViewContentModeScaleAspectFit;
        _thumbImageView.clipsToBounds = YES;
    }
    return _thumbImageView;
}

- (UIView *)playerView{
    if (!_playerView) {
        _playerView = [[UIView alloc] init];
    }
    return _playerView;
}

- (UIButton *)playButton{
    if (!_playButton){
        _playButton = [UIButton buttonWithType:UIButtonTypeCustom];
        [_playButton setImage:[UIImage imageNamed:@"feed_icon_display"]
                     forState:UIControlStateNormal];
        [_playButton addTarget:self action:@selector(playAction)
              forControlEvents:UIControlEventTouchUpInside];
    }
    return _playButton;
}

- (TAVTemplateEditorBottomView *)bottomView {
    if (!_bottomView) {
        _bottomView = [[TAVTemplateEditorBottomView alloc] init];
        _bottomView.backgroundColor = [UIColor tvc_colorWithHexString:@"#2B2C30"];
        _bottomView.delegate = self;
    }
    return _bottomView;
}

- (UIButton *)bgmBtn {
    if (!_bgmBtn) {
        _bgmBtn = [UIButton buttonWithType:UIButtonTypeCustom];
        [_bgmBtn setImage:[UIImage imageNamed:@"icon_sound_default"] forState:UIControlStateNormal];
        _bgmBtn.titleLabel.font = [UIFont systemFontOfSize:12];
        [_bgmBtn setTitleColor:[UIColor whiteColor] forState:UIControlStateNormal];
        _bgmBtn.backgroundColor = [UIColor colorWithWhite:0 alpha:0.6];
        [_bgmBtn addTarget:self action:@selector(bgmBtnClick:) forControlEvents:UIControlEventTouchUpInside];
    }
    return _bgmBtn;
}

#pragma mark - TAVTemplateEditorBottomViewDelegate
- (void)bottomViewDidPlayAction {
    [self playAction];
}

- (void)bottomViewDidSliderStartDragging {
    [self didStartDragging];
}

- (void)bottomViewDidSliderDragging:(float)value {
    [self didChangeTime:value];
}

- (void)bottomViewDidSliderEndDragging {
    [self didEndDragging];
}

@end
