//
//  TAVTemplateVideoSetView.m
//  TAVMagicApp
//
//  Created by jasonggao on 2025/6/3.
//

#import "TAVTemplateVideoSetView.h"
#import <TAVMagicUIKit/UIColor+TAVUtil.h>

static const int frameRate30 = 30;
static const int frameRate25 = 25;
static const int frameRate20 = 20;
static const int frameRate15 = 15;
static const float bppHigh      = 0.2; //高质量
static const float bppNormal    = 0.15;//推荐
static const float bppLow       = 0.1; //低质量

@interface TAVTemplateVideoSetView ()<UITableViewDelegate,UITableViewDataSource>
@property (nonatomic, strong) UITableView *tableView;
@property (nonatomic, strong) UIView *separateLineView;
@property (nonatomic, strong) UIButton *sheetCancelBtn;
@property (nonatomic, assign, readwrite) int videoBitRate; // 平均码率
@property (nonatomic, assign) float bpp;
@property (nonatomic, assign, readwrite) int frameRate; // 帧率
@property (nonatomic, assign, readwrite) TAVVideoSetResolution resolution; // 分辨率
@property (nonatomic, assign, readwrite) CGSize resolutionSize; // 分辨率size
@property (nonatomic, strong) NSArray *titleDataSource;
@property (nonatomic, strong) NSMutableArray *subTitleDataSource;
@property (nonatomic, strong) NSArray *resolutionDataSource;
@property (nonatomic, strong) NSArray *frameDataSource;
@property (nonatomic, strong) NSArray *bitRateDataSource;

@end

@implementation TAVTemplateVideoSetView

- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        [self initDate];
        [self setupUI];
    }
    return self;
}

- (void)initDate {
    self.resolution = TAVVideoSetResolution1080P;
    self.resolutionSize = CGSizeMake(1920, 1080);
    self.frameRate = frameRate30;
    self.bpp = bppNormal;
    self.videoBitRate = [self caculateVideoBitRateWithResolution:TAVVideoSetResolution1080P frameRate:frameRate30 bpp:bppNormal];
    self.resolutionString = @"1080P";
}

- (void)setupUI {
    self.titleLabel.text = @"视频设置";
    [self.rightButton setTitle:@"完成" forState:UIControlStateNormal];
    
    self.titleDataSource = @[@"分辨率", @"帧率", @"码率"];
    self.subTitleDataSource = [[NSMutableArray alloc] initWithArray:@[@"1080P", @"30", @"推荐"]];
    self.resolutionDataSource = @[@"4K", @"1080P", @"720P", @"540P"];
    self.frameDataSource = @[@"30", @"25", @"20", @"15"];
    self.bitRateDataSource = @[@"较高", @"推荐", @"较低"];
    
    [self addSubview:self.tableView];
    
    self.separateLineView = [[UIView alloc] init];
    self.separateLineView.backgroundColor = [UIColor tvc_colorWithHexString:@"#2B2C30"];
    [self addSubview:self.self.separateLineView];
    
    [self addSubview:self.sheetCancelBtn];
}

- (void)layoutSubviews {
    [super layoutSubviews];
    CGFloat originX = 0;
    CGFloat originY = 0;
    CGFloat width = self.bounds.size.width;
    CGFloat height = 168.0;
    if (self.type == TAVVideoSetTypeVideoInfo) {
        height = self.titleDataSource.count * 56;
        originX = 16.0;
        originY = 64.0;
        width = self.bounds.size.width - 2 * originX;
    } else if (self.type == TAVVideoSetTypeResolution) {
        height = self.resolutionDataSource.count * 56;
    } else if (self.type == TAVVideoSetTypeFrame) {
        height = self.frameDataSource.count * 56;
    } else if (self.type == TAVVideoSetTypeBitRate) {
        height = self.bitRateDataSource.count * 56;
    }
    self.tableView.frame = CGRectMake(originX, originY, width, height);
    self.separateLineView.frame = CGRectMake(0, CGRectGetMaxY(self.tableView.frame), width, 7);
    self.sheetCancelBtn.frame = CGRectMake(0, CGRectGetMaxY(self.separateLineView.frame), width, 56 + self.safeAreaInsets.bottom);
}

- (void)setType:(TAVVideoSetType)type {
    _type = type;
    if (self.type == TAVVideoSetTypeVideoInfo) {
        self.titleLabel.hidden = NO;
        self.leftButton.hidden = YES;
        self.rightButton.hidden = NO;
        self.separateLineView.hidden = YES;
        self.sheetCancelBtn.hidden = YES;
    } else {
        self.titleLabel.hidden = YES;
        self.leftButton.hidden = YES;
        self.rightButton.hidden = YES;
        self.separateLineView.hidden = NO;
        self.sheetCancelBtn.hidden = NO;
    }
    [self.tableView reloadData];
}

- (void)showSubVideoSetView:(TAVVideoSetType)type {
    CGFloat height = 0;
    switch (type) {
        case TAVVideoSetTypeResolution:
        case TAVVideoSetTypeFrame:
            height = 5 * 56 + 7 + self.safeAreaInsets.bottom;
            break;
        case TAVVideoSetTypeBitRate:
            height = 4 * 56 + 7 + self.safeAreaInsets.bottom;
            break;
        default:break;
    }
    TAVTemplateVideoSetView *subSetView = [[TAVTemplateVideoSetView alloc] initWithFrame:CGRectMake(0, self.superview.bounds.size.height - height, self.superview.bounds.size.width, height)];
    subSetView.type = type;
    __weak typeof(self)weakSelf = self;
    subSetView.ResolutionChange = ^(TAVVideoSetResolution resolution, NSInteger index) {
        weakSelf.resolution = resolution;
        weakSelf.videoBitRate = [weakSelf caculateVideoBitRateWithResolution:weakSelf.resolution frameRate:weakSelf.frameRate bpp:weakSelf.bpp];
        [weakSelf.subTitleDataSource replaceObjectAtIndex:0 withObject:weakSelf.resolutionDataSource[index]];
        [weakSelf.tableView reloadData];
    };
    subSetView.FrameRateChange = ^(int frameRate, NSInteger index) {
        weakSelf.frameRate = frameRate;
        weakSelf.videoBitRate = [weakSelf caculateVideoBitRateWithResolution:weakSelf.resolution frameRate:weakSelf.frameRate bpp:weakSelf.bpp];
        [weakSelf.subTitleDataSource replaceObjectAtIndex:1 withObject:weakSelf.frameDataSource[index]];
        [weakSelf.tableView reloadData];
    };
    subSetView.BitRateChange = ^(float bpp, NSInteger index) {
        weakSelf.videoBitRate = [weakSelf caculateVideoBitRateWithResolution:weakSelf.resolution frameRate:weakSelf.frameRate bpp:bpp];
        [weakSelf.subTitleDataSource replaceObjectAtIndex:2 withObject:weakSelf.bitRateDataSource[index]];
        [weakSelf.tableView reloadData];
    };
    [self.superview addSubview:subSetView];
    [subSetView showActionSheet];
}

- (int)caculateVideoBitRateWithResolution:(TAVVideoSetResolution)resolution frameRate:(int)frameRate bpp:(float)bpp {
    //码率(bps)=宽度×高度×帧率×bpp
    CGSize size = CGSizeZero;
    switch (resolution) {
        case TAVVideoSetResolution4K:
            size = CGSizeMake(3840, 2160);
            self.resolutionString = @"4K";
            break;
        case TAVVideoSetResolution1080P:
            size = CGSizeMake(1920, 1080);
            self.resolutionString = @"1080P";
            break;
        case TAVVideoSetResolution720P:
            size = CGSizeMake(1280, 720);
            self.resolutionString = @"720P";
            break;
        case TAVVideoSetResolution540P:
            size = CGSizeMake(920, 540);
            self.resolutionString = @"540P";
            break;
        default:
            size = CGSizeMake(1920, 1080);
            break;
    }
    self.resolutionSize = size;
    return ceilf(size.width * size.height * frameRate * bpp);
}

- (void)sheetCancelClick:(UIButton *)sender {
    [self hideActionSheet];
}

#pragma mark - Lazy
- (UITableView *)tableView {
    if (!_tableView) {
        _tableView = [[UITableView alloc] initWithFrame:CGRectZero style:UITableViewStylePlain];
        _tableView.backgroundColor = [UIColor tvc_colorWithHexString:@"#3A3C42"];
        _tableView.delegate = self;
        _tableView.dataSource = self;
        _tableView.showsVerticalScrollIndicator = NO;
        _tableView.scrollEnabled = NO;
        _tableView.layer.cornerRadius = 8.0;
        _tableView.separatorStyle = UITableViewCellSeparatorStyleSingleLine;
        _tableView.separatorColor = [UIColor tvc_colorWithHexString:@"#3A3C42"];
        _tableView.separatorInset = UIEdgeInsetsMake(0, 12, 0, 12);
        [_tableView registerClass:[TAVVideoSetCell class] forCellReuseIdentifier:NSStringFromClass(TAVVideoSetCell.class)];
    }
    return _tableView;
}

- (UIButton *)sheetCancelBtn {
    if (!_sheetCancelBtn) {
        _sheetCancelBtn = [UIButton buttonWithType:UIButtonTypeCustom];
        [_sheetCancelBtn setTitle:@"取消" forState:UIControlStateNormal];
        [_sheetCancelBtn setTitleColor:[UIColor whiteColor] forState:UIControlStateNormal];
        _sheetCancelBtn.titleLabel.font = [UIFont systemFontOfSize:16];
        _sheetCancelBtn.titleEdgeInsets = UIEdgeInsetsMake(-10, 0, 10, 0);
        _sheetCancelBtn.backgroundColor = [UIColor tvc_colorWithHexString:@"#1F2024"];
        [_sheetCancelBtn addTarget:self action:@selector(sheetCancelClick:) forControlEvents:UIControlEventTouchUpInside];
    }
    return _sheetCancelBtn;
}

#pragma mark - UITableViewDataSource
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section {
    if (self.type == TAVVideoSetTypeVideoInfo) {
        return self.titleDataSource.count;
    } else if (self.type == TAVVideoSetTypeResolution) {
        return self.resolutionDataSource.count;
    } else if (self.type == TAVVideoSetTypeFrame) {
        return self.frameDataSource.count;
    } else if (self.type == TAVVideoSetTypeBitRate) {
        return self.bitRateDataSource.count;
    }
    return 0;
}

- (CGFloat)tableView:(UITableView *)tableView heightForRowAtIndexPath:(NSIndexPath *)indexPath {
    return 56.0;
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath {
    TAVVideoSetCell *cell = [tableView dequeueReusableCellWithIdentifier:NSStringFromClass(TAVVideoSetCell.class)];
    cell.cellType = self.type;
    cell.selectionStyle = UITableViewCellSelectionStyleNone;
    if (self.type == TAVVideoSetTypeVideoInfo) {
        cell.titleLabel.text = self.titleDataSource[indexPath.row];
        cell.subTitleLabel.text = self.subTitleDataSource[indexPath.row];
    } else if (self.type == TAVVideoSetTypeResolution) {
        cell.titleLabel.text = self.resolutionDataSource[indexPath.row];
    } else if (self.type == TAVVideoSetTypeFrame) {
        cell.titleLabel.text = self.frameDataSource[indexPath.row];
    } else if (self.type == TAVVideoSetTypeBitRate) {
        cell.titleLabel.text = self.bitRateDataSource[indexPath.row];
    }
    return cell;
}

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath {
    TAVVideoSetType type;
    if (self.type == TAVVideoSetTypeVideoInfo) {
        switch (indexPath.row) {
            case 0:type = TAVVideoSetTypeResolution;break;
            case 1:type = TAVVideoSetTypeFrame;break;
            case 2:type = TAVVideoSetTypeBitRate;break;
            default:type = TAVVideoSetTypeResolution;break;
        }
        [self showSubVideoSetView:type];
    } else if (self.type == TAVVideoSetTypeResolution) {
        TAVVideoSetResolution resolution = TAVVideoSetResolution1080P;
        switch (indexPath.row) {
            case 0:resolution = TAVVideoSetResolution4K;break;
            case 1:resolution = TAVVideoSetResolution1080P;break;
            case 2:resolution = TAVVideoSetResolution720P;break;
            case 3:resolution = TAVVideoSetResolution540P;break;
        }
        if (self.ResolutionChange) {
            self.ResolutionChange(resolution, indexPath.row);
        }
        [self hideActionSheet];
    } else if (self.type == TAVVideoSetTypeFrame) {
        int frameRate = frameRate30;
        switch (indexPath.row) {
            case 0:frameRate = frameRate30;break;
            case 1:frameRate = frameRate25;break;
            case 2:frameRate = frameRate20;break;
            case 3:frameRate = frameRate15;break;
        }
        if (self.FrameRateChange) {
            self.FrameRateChange(frameRate, indexPath.row);
        }
        [self hideActionSheet];
    } else if (self.type == TAVVideoSetTypeBitRate) {
        float bbp = bppNormal;
        switch (indexPath.row) {
            case 0:bbp = bppHigh;break;
            case 1:bbp = bppNormal;break;
            case 2:bbp = bppLow;break;
        }
        if (self.BitRateChange) {
            self.BitRateChange(bbp, indexPath.row);
        }
        [self hideActionSheet];
    }
}

@end
