//
//  TAVRecordViewController.h
//  TAVMagicApp
//
//  Created by v_zwtzzhou on 2025/5/13.
//

#import <UIKit/UIKit.h>
#import <AVFoundation//AVAsset.h>

NS_ASSUME_NONNULL_BEGIN

typedef NS_ENUM(NSInteger, TAVVideoRecordSpeedMode) {
    TAVVideoRecordSpeedMode_VerySlow,
    TAVVideoRecordSpeedMode_Slow,
    TAVVideoRecordSpeedMode_Standard,
    TAVVideoRecordSpeedMode_Quick,
    TAVVideoRecordSpeedMode_VeryQuick,
};

@class TAVRecordViewController;
@protocol TAVRecordStatusDelegate <NSObject>
@optional
/**
 @brief 开始录制
 */
- (void)onRecordStarted:(TAVRecordViewController *)recordViewController;

/**
 @brief 暂停录制
 */
- (void)onRecordPaused:(TAVRecordViewController *)recordViewController;

/**
 @brief 恢复录制
 */
- (void)onRecordResumed:(TAVRecordViewController *)recordViewController;

/**
 @brief 录制结束
 */
- (void)onRecordFinished:(TAVRecordViewController *)recordViewController
                  videoPath:(NSString *)path;

/**
 @brief 录制中断
 */
- (void)onRecordInterrupted:(TAVRecordViewController *)recordViewController;

/**
 @brief 录制进度回掉
 */
- (void)onRecordProgress:(CGFloat)progress millSecond:(NSInteger)milliSecond;

@end

@protocol TAVRecordProcessDelegate <NSObject>
@optional
/**
 * 在 OpenGL 线程中回调，在这里可以进行采集图像的二次处理
 * @param texture    纹理 ID。
 * @param width      纹理的宽度。
 * @param height     纹理的高度。
 * @return           返回给 SDK 的纹理。
 * 说明：SDK 回调出来的纹理类型是 GL_TEXTURE_2D，接口返回给 SDK 的纹理类型也必须是 GL_TEXTURE_2D。
 * 该回调在 SDK 美颜之后. 纹理格式为 GL_RGBA。
 */
- (GLuint)onPreProcessTexture:(GLuint)texture width:(CGFloat)width height:(CGFloat)height;

/**
 * 在 OpenGL 线程中回调，可以在这里释放创建的 OpenGL 资源
 */
- (void)onTextureDestoryed;

/**
 * 人脸数据回调（增值版且启用了 pitu 模块才有效）
 * @param points 人脸坐标。
 *  说明：开启 pitu 模块必须是打开动效或大眼瘦脸。此回调在 onPreProcessTexture:width:height: 之前。
 */
- (void)onDetectFacePoints:(NSArray *)points;
@end


@interface TAVRecordViewController : UIViewController

@property (nonatomic, weak)id<TAVRecordStatusDelegate>recordStatusDelegate;
@property (nonatomic, weak) id<TAVRecordProcessDelegate> videoProcessDelegate;
@property (nonatomic, assign) CGFloat minDuration;
@property (nonatomic, assign) CGFloat maxDuration;

/**
 @brief 视频/图片保存路径
 */
@property (nonatomic, copy) NSString *exportPath;

/**
 @brief 是否开启闪光灯
 */
@property (nonatomic, assign) BOOL isFlash;
@property (nonatomic, copy) void(^flashChanged)(BOOL isOpen);

/**
 @brief 是否使用前摄像头
 */
@property (nonatomic, assign) BOOL isFrontCamera;
@property (nonatomic, copy) void(^cameraChanged)(BOOL isFrontCamera);

/**
 @brief 是否使用麦克风
 */
@property (nonatomic, assign) BOOL isMic;
@property (nonatomic, copy) void(^micChanged)(BOOL isMic);

/**
 @brief 拍摄速度
 */
@property (nonatomic, assign) TAVVideoRecordSpeedMode recordSpeed;
@property (nonatomic, copy) void(^recordSpeedChanged)(TAVVideoRecordSpeedMode recordSpeed);

/**
 @brief 是否在录制
 */
@property (nonatomic, assign, readonly)BOOL isRecording;

/**
 @brief 开始录制
 */
- (BOOL)resumeRecordWithError:(NSError * _Nullable * _Nullable)error;

/**
 @brief 暂停录制
 */
- (void)pauseRecord;

/**
 @brief 丢弃录制
 */
- (void)discardRecord;

/**
 @brief 结束录制
 */
- (void)stopRecord;

/**
 @brief 取消录制
 */
- (void)cancelRecord;

/**
 @brief 拍照
 */
- (void)takePhoto:( void(^)(UIImage * _Nullable image,  NSString * _Nullable path))callback;

/**
 @brief 设置背景音乐
 */
- (BOOL)setBGM:(NSString *)path;
- (BOOL)setBGMAsset:(AVAsset *)asset;
- (void)removeBGM;
- (void)setBGMValidCallback:(void(^)(BOOL isValid))callback;
- (BOOL)isUsingBGM;

@end

NS_ASSUME_NONNULL_END
