//
//  TAVRecordViewController.m
//  TAVMagicApp
//
//  Created by v_zwtzzhou on 2025/5/13.
//

#import "TAVRecordViewController.h"
#import "TXUGCRecord.h"
#import "TXUGCRecordTypeDef.h"

static TXVideoRecordSpeed bridgeSpeedMode(TAVVideoRecordSpeedMode speedMode){
    switch (speedMode) {
        case TAVVideoRecordSpeedMode_VerySlow: return VIDEO_RECORD_SPEED_SLOWEST;  break;
        case TAVVideoRecordSpeedMode_Slow: return VIDEO_RECORD_SPEED_SLOW;  break;
        case TAVVideoRecordSpeedMode_Standard: return VIDEO_RECORD_SPEED_NOMAL;  break;
        case TAVVideoRecordSpeedMode_Quick: return VIDEO_RECORD_SPEED_FAST;  break;
        case TAVVideoRecordSpeedMode_VeryQuick: return VIDEO_RECORD_SPEED_FASTEST;  break;
        default: return VIDEO_RECORD_SPEED_NOMAL; break;
    }
}

@interface TAVRecordViewController ()<TXUGCRecordListener, TXVideoCustomProcessDelegate>{
    BOOL _isCameraPreviewOn;    // 是否在进行预览
    BOOL _isStartedRecord;      // 已经开始录制
    BOOL _isCompletingRecord;   // 是否正在结束录制
    BOOL _isUsingBgm;           // 是否使用bgm
    CGFloat _bgmDuration;       // bgm时长
    BOOL _isBgmPlayed;          // bgm已经播放
    NSInteger _recordTime;      // 录制总时间,单位毫秒
}
@property (nonatomic, strong)TXUGCCustomConfig *config;
@property (nonatomic, copy) void(^BGMValidCallback)(BOOL isValid);
@end

@implementation TAVRecordViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    // Do any additional setup after loading the view.
    [self settingDefaultProperty];
    [self settingConfig];
    [TXUGCRecord shareInstance].recordDelegate = self;
    [TXUGCRecord shareInstance].videoProcessDelegate = self;
}

- (void)viewWillAppear:(BOOL)animated{
    [super viewWillAppear:animated];
    [self startCameraPreview];
}

- (void)viewDidDisappear:(BOOL)animated{
    [super viewDidDisappear:animated];
    [self stopCameraPreview];
}

// 退后台停止渲染
- (void)viewWillResignActive:(NSNotification *)noti {
}

// 后台返回恢复动效
- (void)viewDidBecomeActive:(NSNotification *)noti {
    // 恢复动效
}

#pragma mark - Common
- (void)settingDefaultProperty{
    _isCameraPreviewOn = NO;
    _isStartedRecord = NO;
    _isCompletingRecord = NO;
    _recordTime = 0;
    
    _isFrontCamera = YES;
    _isFlash = NO;
}

- (void)settingConfig{
    TXUGCCustomConfig *config = [[TXUGCCustomConfig alloc] init];
    config.videoResolution = VIDEO_RESOLUTION_720_1280;
    config.videoFPS = 30;
    config.videoBitratePIN = 9600;
    config.enableAEC = YES;
    config.minDuration = self.minDuration;
    config.maxDuration = self.maxDuration;
    config.audioSampleRate = AUDIO_SAMPLERATE_32000;
    config.frontCamera = YES;
    
    self.config = config;
}

- (void)setMinDuration:(CGFloat)minDuration{
    _minDuration = minDuration;
    self.config.minDuration = minDuration;
}

- (void)setMaxDuration:(CGFloat)maxDuration{
    _maxDuration = maxDuration;
    self.config.maxDuration = maxDuration;
}

- (void)setIsFlash:(BOOL)isFlash{
    if (_isCameraPreviewOn){
        if ([[TXUGCRecord shareInstance] toggleTorch:isFlash]){
            _isFlash = isFlash;
        }
    } else {
        _isFlash = isFlash;
    }
    
    if (self.flashChanged){
        self.flashChanged(_isFlash);
    }
}

- (void)setIsFrontCamera:(BOOL)isFrontCamera{
    if (_isCameraPreviewOn){
        if ([[TXUGCRecord shareInstance] switchCamera:isFrontCamera]) {
            _isFrontCamera = isFrontCamera;
        }
    }
    _isFrontCamera = isFrontCamera;
    
    if (self.cameraChanged){
        self.cameraChanged(_isFrontCamera);
    }
    
    if (_isFrontCamera && _isFlash){
        self.isFlash = NO;
    }
}

- (void)setIsMic:(BOOL)isMic{
    if ( [[TXUGCRecord shareInstance] setMicVolume:isMic ? 1 : 0]) {
        _isMic = isMic;
    };
    
    if (self.micChanged){
        self.micChanged(_isMic);
    }
    if (_isMic){
        [self removeBGM];
    }
}

- (void)setRecordSpeed:(TAVVideoRecordSpeedMode)recordSpeed{
    _recordSpeed = recordSpeed;
    
    [[TXUGCRecord shareInstance] setRecordSpeed:bridgeSpeedMode(recordSpeed)];
    
    if (self.recordSpeedChanged){
        self.recordSpeedChanged(_recordSpeed);
    }
}

- (void)setBGMValidCallback:(void(^)(BOOL isValid))callback{
    _BGMValidCallback = [callback copy];
}

- (BOOL)isUsingBGM{
    return _isBgmPlayed;
}

#pragma mark - Record
#define RECORD_RETURN_ERROR(_condition, _code, _msg) \
if (_condition){\
    if (error){\
        *error = [NSError errorWithDomain:@"TAVCameraCaptureError" code:_code userInfo:nil];\
    }\
    return NO;\
}
- (int)startRecordWithError:(NSError * _Nullable * _Nullable)error {
    if (_isStartedRecord){
        return 0;
    }
    [self startCameraPreview];
    NSString *filePath = [self.exportPath stringByAppendingPathComponent:[NSUUID UUID].UUIDString];
    int result = [[TXUGCRecord shareInstance] startRecord:[filePath stringByAppendingString:@".mp4"]
                                                coverPath:[filePath stringByAppendingString:@".png"]];
    
    if (0 == result){
        _isStartedRecord = YES;
    } else if (error) {
        *error = [NSError errorWithDomain:@"TAVRecordError" code:result userInfo:nil];
    }
    return result;
}

- (BOOL)resumeRecordWithError:(NSError * _Nullable * _Nullable)error {
    RECORD_RETURN_ERROR((!self.exportPath || (self.exportPath.length == 0)), 1, "缺少视频保存路径");
    RECORD_RETURN_ERROR(_isCompletingRecord, 2, @"视频已经结束录制");
    
    // recording
    if (_isStartedRecord) {
        [[TXUGCRecord shareInstance] resumeRecord];
    } else {
        int result = [self startRecordWithError:error];
        RECORD_RETURN_ERROR((0 != result), result, "启动失败");
    }
    _isRecording = YES;

    // BGM
    if (_isUsingBgm) {
        if (_isBgmPlayed) {
            [[TXUGCRecord shareInstance] resumeBGM];
        } else {
            _isBgmPlayed = [[TXUGCRecord shareInstance] playBGMFromTime:0 toTime:_bgmDuration withBeginNotify:^(NSInteger errCode) {
            } withProgressNotify:^(NSInteger progressMS, NSInteger durationMS) {
            } andCompleteNotify:^(NSInteger errCode) {
            }];
        }
    }
    
    if ([self.recordStatusDelegate respondsToSelector:@selector(onRecordStarted:)]){
        [self.recordStatusDelegate onRecordStarted:self];
    }
    return YES;
}

- (void)pauseRecord{
    if (!_isRecording){
        return;
    }
    _isRecording = NO;
    
    if (_isBgmPlayed){
        [[TXUGCRecord shareInstance] pauseBGM];
    }
    
    [[TXUGCRecord shareInstance] pauseRecord:nil];
    if ([self.recordStatusDelegate respondsToSelector:@selector(onRecordPaused:)]){
        [self.recordStatusDelegate onRecordPaused:self];
    }
}

- (void)discardRecord {
    [[TXUGCRecord shareInstance].partsManager deleteAllParts];
    _recordTime = 0;
    if ([self.recordStatusDelegate respondsToSelector:@selector(onRecordProgress:millSecond:)]){
        [self.recordStatusDelegate onRecordProgress:0 millSecond:0];
    }
}

- (void)stopRecord {
    _isCompletingRecord = YES;
    _isRecording = NO;
    [[TXUGCRecord shareInstance] stopRecord];
    if (_isBgmPlayed){
        [[TXUGCRecord shareInstance] stopBGM];
    }
    _isBgmPlayed = NO;
    // 结束回调在-onRecordComplete:
}


- (void)cancelRecord {
    if (_isRecording){
        _isRecording = NO;
        [[TXUGCRecord shareInstance] pauseRecord:^{
            [[TXUGCRecord shareInstance].partsManager deleteAllParts];
            [[TXUGCRecord shareInstance] stopRecord];
        }];
    }
}

-(void)stopRecordAndComplete:(void(^)(int))completion
{
    if (_isCompletingRecord) {
        return;
    }
    _isCompletingRecord = YES;
}

- (void)takePhoto:(void(^)(UIImage *image, NSString *path))callback{
    __block NSString *filePath = [self.exportPath stringByAppendingPathComponent:[NSUUID UUID].UUIDString];
    filePath = [filePath stringByAppendingString:@".png"];
    [[TXUGCRecord shareInstance] snapshot:^(UIImage *image) {
        if (!callback){
            return;
        }
        
        if (image){
            NSData *imageData = UIImagePNGRepresentation(image);
            NSURL *fileURL = [NSURL fileURLWithPath:filePath];
            BOOL succ = [imageData writeToURL:fileURL options:NSDataWritingAtomic error:nil];
            if (!succ){
                filePath = nil;
            }
        } else {
            filePath = nil;
        }
        
        dispatch_async(dispatch_get_main_queue(), ^{
            callback(image, filePath);
        });
    }];
}


- (BOOL)setBGM:(NSString *)path{
    if (_isBgmPlayed){
        return NO;
    }
    _bgmDuration = [[TXUGCRecord shareInstance] setBGM:path];
    [self resetUsingBGM:_bgmDuration > 0];
    return _isUsingBgm;
}
- (BOOL)setBGMAsset:(AVAsset *)asset {
    if (_isBgmPlayed){
        return NO;
    }
    _bgmDuration = [[TXUGCRecord shareInstance] setBGMAsset:asset];
    [self resetUsingBGM:_bgmDuration > 0];
    return _isUsingBgm;
}

- (BOOL)resetUsingBGM:(BOOL)usingBGM{
    _isUsingBgm = usingBGM;
    if (_isUsingBgm){
        [[TXUGCRecord shareInstance] setBGMLoop:YES];
        self.isMic = NO;
    }
    
    if (self.BGMValidCallback){
        self.BGMValidCallback(_isUsingBgm);
    }
    return _isUsingBgm;
}

- (void)removeBGM {
    if (!_isUsingBgm){
        return;
    }
    _isUsingBgm = NO;
    
    if (_isBgmPlayed && [[TXUGCRecord shareInstance] stopBGM]){
        _isBgmPlayed = NO;
    }
    
    if (self.BGMValidCallback){
        self.BGMValidCallback(NO);
    }
}

#pragma mark - Priview
-(void)startCameraPreview {
    if (_isCameraPreviewOn){
        return;
    }
    _isCameraPreviewOn = YES;
    [[TXUGCRecord shareInstance] startCameraCustom:self.config preview:self.view];
    [[TXUGCRecord shareInstance] setVideoRenderMode:VIDEO_RENDER_MODE_FULL_FILL_SCREEN];
    
    // 刷新一下配置并进行回调
    self.isFlash = _isFlash;
    self.isFrontCamera = _isFrontCamera;
}

- (void)stopCameraPreview{
    if (!_isCameraPreviewOn) {
        return;
    }
    _isCameraPreviewOn = NO;
    [[TXUGCRecord shareInstance] stopCameraPreview];
}

#pragma mark - TXUGCRecordListener
- (void)onRecordProgress:(NSInteger)milliSecond{
    _recordTime =  milliSecond;
    CGFloat t = _recordTime / 1000.0;
    CGFloat progress = t / _config.maxDuration;
    if ([self.recordStatusDelegate respondsToSelector:@selector(onRecordProgress:millSecond:)]){
        [self.recordStatusDelegate onRecordProgress:progress millSecond:_recordTime];
    }
    
    BOOL shouldPause = (t >= _config.maxDuration);
    if (shouldPause) {
        [self pauseRecord];
    }
}

- (void)onRecordComplete:(TXUGCRecordResult*)result{
    if (UGC_RECORD_RESULT_FAILED == result.retCode){
        return;
    } else if (UGC_RECORD_RESULT_OK_INTERRUPT == result.retCode){
        if ([self.recordStatusDelegate respondsToSelector:@selector(onRecordInterrupted:)]){
            [self.recordStatusDelegate onRecordInterrupted:self];
        }
        // 中断处理
        _isStartedRecord = NO;
    } else {
        
        NSString *videoPath = result.videoPath;
        if ([self.recordStatusDelegate respondsToSelector:@selector(onRecordFinished:videoPath:)]){
            [self.recordStatusDelegate onRecordFinished:self videoPath:videoPath];
        }
        
        if (_isCompletingRecord){
            [[TXUGCRecord shareInstance].partsManager deleteAllParts];
            _isCompletingRecord = NO;
        }
        _isStartedRecord = NO;
    }
}

#pragma mark - TXVideoCustomProcessDelegate
- (GLuint)onPreProcessTexture:(GLuint)texture width:(CGFloat)width height:(CGFloat)height {
    if ([self.videoProcessDelegate respondsToSelector:@selector(onPreProcessTexture:width:height:)]){
//        static GLuint res = 1;
//        return res;
        return [self.videoProcessDelegate onPreProcessTexture:texture width:width height:height];
    }
    return texture;
}

- (void)onTextureDestoryed {
    if ([self.videoProcessDelegate respondsToSelector:@selector(onTextureDestoryed)]){
        return [self.videoProcessDelegate onTextureDestoryed];
    }
}
- (void)onDetectFacePoints:(NSArray *)points {
    if ([self.videoProcessDelegate respondsToSelector:@selector(onDetectFacePoints:)]){
        return [self.videoProcessDelegate onDetectFacePoints:points];
    }
}

@end
