//
//  TAVEditableView.h
//  TAVStickerKit
//
//  Created by Victor Tian on 2019/3/7.
//  Copyright © 2019 Tencent Inc. All rights reserved.
//

#import <UIKit/UIKit.h>

NS_ASSUME_NONNULL_BEGIN

/**
 Which events should be enabled.
 */
typedef NS_OPTIONS(NSInteger, TAVEditableViewEventType) {
    /** Disable all gesture. */
    TAVEditableViewEventTypeNone        = 0 ,
    /** Enable touch gesture. */
    TAVEditableViewEventTypeTouch       = 1 << 0 ,
    /** Enable tap gesture. */
    TAVEditableViewEventTypeTap         = 1 << 1 ,
    /** Enable pan gesture. */
    TAVEditableViewEventTypePan         = 1 << 2 ,
    /** Enable pinch gesture. */
    TAVEditableViewEventTypePinch       = 1 << 3 ,
    /** Enable rotation gesture. */
    TAVEditableViewEventTypeRotation    = 1 << 4 ,
    /** Enable transform gesture. */
    TAVEditableViewEventTypeTransform   = 1 << 5 ,
    /** Enable rectangular gesture. */
    TAVEditableViewEventTypeRectangular = 1 << 6 ,
    /** Enable all internal gesture. */
    TAVEditableViewEventTypeAll         = TAVEditableViewEventTypeTouch | TAVEditableViewEventTypeTap | TAVEditableViewEventTypePan |
                                          TAVEditableViewEventTypePinch | TAVEditableViewEventTypeRotation,
};

@class TAVEditableView;
@protocol TAVEditableViewDelegate <NSObject>

@optional
/**
 Tells the delegate that current editable-view has been clicked.
 
 @param editableView The editable-view object.
 */
- (void)editableViewHasBeenClicked:(TAVEditableView *)editableView;
/**
 Tells the delegate that the position of editable-view has changed.

 @param editableView The editable-view object.
 @param position Current position of the editable-view object.
 */
- (void)editableView:(TAVEditableView *)editableView
   positionDidChange:(CGPoint)position;
/**
 Tells the delegate that the position of editable-view will change.
 
 @param editableView The editable-view object.
 @param position Current position of the editable-view object.
 */
- (CGPoint)editableView:(TAVEditableView *)editableView
     positionWillChange:(CGPoint)position;
/**
 Tells the delegate that the rotation of editable-view has changed.

 @param editableView The editable-view object.
 @param rotation Current rotation of the editable-view object.
 */
- (void)editableView:(TAVEditableView *)editableView
   rotationDidChange:(CGFloat)rotation;
/**
 Tells the delegate that the rotation of editable-view will change.
 
 @param editableView The editable-view object.
 @param rotation Current rotation of the editable-view object.
 */
- (CGFloat)editableView:(TAVEditableView *)editableView
     rotationWillChange:(CGFloat)rotation;
/**
 Tells the delegate that the scale of editable-view has changed.

 @param editableView The editable-view object.
 @param scale Current scale of the editable-view object.
 */
- (void)editableView:(TAVEditableView *)editableView
      scaleDidChange:(CGFloat)scale;
/**
 Tells the delegate that the scale of editable-view will change.
 
 @param editableView The editable-view object.
 @param scale Current scale of the editable-view object.
 */
- (CGFloat)editableView:(TAVEditableView *)editableView
        scaleWillChange:(CGFloat)scale;
/**
 Tells the delegate that the editable-view has responded the specified gesture.

 @param editableView The editable-view object.
 @param eventType The gesture type.
 */
- (void)editableView:(TAVEditableView *)editableView
     didBeginEditing:(TAVEditableViewEventType)eventType;
/**
 Tells the delegate that the editable-view did end responding the specified gesture.
 
 @param editableView The editable-view object.
 @param eventType The gesture type.
 */
- (void)editableView:(TAVEditableView *)editableView
       didEndEditing:(TAVEditableViewEventType)eventType;

@end

@interface TAVEditableView : UIView
@property (nonatomic, assign) CGPoint rotationCenter; // 旋转中心点坐标（父视图坐标系）
/** limit area, value in [0, 1.0]} */
@property (nonatomic, assign) CGRect limitArea;
/** minimum scale, value in [0, 1.0], default 0 */
@property (nonatomic, assign) CGFloat minimumScale;
/** maximum scale, value in [1, n], default CGFLOAT_MAX */
@property (nonatomic, assign) CGFloat maximumScale;
@property (nonatomic, assign) CGFloat currentScale;
/** The custom distance that the content view is inset from the safe area. */
@property (nonatomic, assign) UIEdgeInsets contentInset;
@property (nonatomic, assign, getter=isEditing) BOOL editing;
/**
 Event should be responsed: tap, pan, pinch, rotation.
 @warning if not contains touch, all gestures are disabled.
 */
@property (nonatomic, assign) TAVEditableViewEventType eventType;
/** Event is responding, pan, pinch, rotation, transform */
@property (nonatomic, assign, readonly) TAVEditableViewEventType currentEvent;
@property (nonatomic, weak) id<TAVEditableViewDelegate> delegate;

- (void)handleTapRecognizer:(UITapGestureRecognizer *)recognizer;
- (void)handlePanRecognizer:(UIPanGestureRecognizer *)recognizer;
- (void)handlePinchRecognizer:(UIPinchGestureRecognizer *)recognizer;
- (void)handleRotationRecognizer:(UIRotationGestureRecognizer *)recognizer;
- (void)handleTransformRecognizer:(UIPanGestureRecognizer *)recognizer;
- (void)handleRectangularRecognizer:(UIPanGestureRecognizer *)recognizer;
- (void)handleLongPressRecognizer:(UILongPressGestureRecognizer *)recognizer;

/** override this func to revise position */
- (CGPoint)positionWillChange:(CGPoint)position;
- (CGFloat)rotationWillChange:(CGFloat)rotation;
- (CGFloat)scaleWillChange:(CGFloat)scale;

- (CGPoint)reviseCenter:(CGPoint)center validRect:(CGRect)validRect;

@end

NS_ASSUME_NONNULL_END
