//
//  TAVMagic.h
//  TAVMagic
//
//  Created by chavezchen on 2023/11/27.
//

#import <Foundation/Foundation.h>
#import <TAVMagic/TAVEditorDefinitions.h>

@class TAVMediaStickerItem;
@class TAVClipInfo;
@class TAVMediaAssetCropInfo;

@protocol TAVStickerContextObserver;
@protocol ITAVStickerManager;
@protocol ITAVPipManager;
@protocol ITAVFrameProvider;
@protocol ITAVDraftManager;

@interface TAVEditor : NSObject

/// 初始化sdk
/// - Parameters:
///   - renderSize: 画面渲染尺寸，根据宽高比缩放。传CGSizeZore则自动适应渲染尺寸
///   - assetsPath: 模型资源路径
- (instancetype)initWithRenderSize:(CGSize)renderSize
                        assetsPath:(NSString *)assetsPath;
+ (instancetype)editorWithRenderSize:(CGSize)renderSize
                          assetsPath:(NSString *)assetsPath;
/// 释放sdk
- (void)deinit;

/// fork editor
- (instancetype)fork;

+ (void)setLogLevel:(TAVLogLevel)logLevel listener:(id<TAVLogListener>)listener;

#pragma - version
/// 获取SDK当前版本号
+ (NSString *)getSdkVersion;

#pragma - resouce input
/// 向末尾添加视频资源
/// - Parameter path: 视频素材路径
/// - return 返回成功添加后的索引
- (int)addVideoClip:(NSString *)path;

/// 指定位置添加视频资源
/// - Parameters:
///   - paths: 视频资源路径数组
///   - index: 添加的位置
- (int)addVideoClips:(NSArray *)paths index:(int)index;

/// 相末尾添加图片资源
/// - Parameters:
///   - path: 图片资源路径
///   - duration: 图片素材展示的时间，传kCMTimeZero为默认3s
- (int)addImageClip:(NSString *)path duration:(CMTime)duration;

/// 在指定位置添加图片资源
/// - Parameters:
///   - path: 图片资源路径
///   - index: 插入位置
///   - durations: 展示时长，单位为秒，传@(0)为默认3s
- (int)addImageClips:(NSArray<NSString *> *)paths index:(int)index durations:(NSArray<NSNumber *> *)durations;

/// 获取当前正在编辑的资源数量
/// @return 资源数量
- (int)getClipCount;

/// 获取当前使用资源信息
- (NSArray<TAVClipInfo *> *)getClipInfos;

/// 移除指定位置的视频或者图片资源
/// - Parameter index: 资源下标
- (void)removeClip:(int)index;

/// 删除所有资源(视频/图片/模板)片段
- (void)removeAllClip;

/// 清空所有资源(video & bgm)和所有特效(sticker & filter & motion)
- (void)clearAll;

/// 修改指定资源的时长
/// - Parameters:
///   - timeRange: 资源播放区间
///   - index: 资源下标
- (void)setClipRange:(CMTimeRange)timeRange index:(int)index;

/// 修改指定资源的速率
/// - Parameters:
///   - speed: 速率 (0,3]
///   - index: 资源下标
- (void)setClipSpeed:(CGFloat)speed index:(int)index;

/// 调整资源位置
/// - Parameters:
///   - oldIndex: 调整前下标
///   - newIndex: 调整后下标
- (BOOL)updateClipIndex:(int)oldIndex toIndex:(int)newIndex;

/// 在改变资源/BGM后需要调用此接口重新生成播放资源
/// 该方法逻辑：1、同步生成播放器资源。2、播放器绑定资源。3、还原状态（异步）：播放状态、播放时时间。
/// 由于该方法存在异步状态，调用完此方法后立马执行播放器的获取时间/是否正在播放等操作可能会出错，需要在第二个参数的completion内再获取。
- (void)flushImmediately;
- (void)flushImmediatelyCompletion:(void (^)(BOOL finished))completion;

/// flushImmediately分别执行以下两步，因为flushImmediately是耗时操作，主要耗时在preparePlayerItem方法内。
/// 故将方法单独抽离出来，供业务更精细化处理耗时问题,可以分别调用以下两个方法达成与flushImmediately相同的操作
- (AVPlayerItem *)preparePlayerItem;
- (void)bindPlayerItem:(AVPlayerItem *)playerItem completion:(void (^)(BOOL finished))completion;

/// 设置指定资源背景音量
/// - Parameters:
///   - volume: 音量0-1f
///   - index: 资源下标
- (void)setClipVolume:(float)volume index:(int)index;

/// 设置所有资源背景音量
/// - Parameters:
///   - volume: 音量0-1f
- (void)setAllClipsVolume:(float)volume;

/// 获取对应下标资源音量
/// - Parameter index: 资源
- (float)getClipVolume:(int)index;

/// 设置视频是否倒放 todo
- (void)setReverse:(BOOL)isReverse;

/// 设置视频旋转角度 按照顺时针旋转
/// - Parameter rotation: 顺时针旋转角度 0-360
- (void)setClipsRotation:(int)rotation;

/// 根据下标设置视频旋转角度
/// - Parameters:
///   - rotation: 顺时针旋转角度 0-360
///   - index: 资源下标
- (void)setClipRotation:(int)rotation index:(int)index;
/// 设置资源形变
- (void)setClipTransform:(CGAffineTransform)transform;
- (void)setClipTransform:(CGAffineTransform)transform index:(int)index;
- (void)setClipSize:(CGSize)size;
- (void)setClipSize:(CGSize)size index:(int)index;
- (void)setCropInfo:(TAVMediaAssetCropInfo *)cropInfo index:(int)index;

#pragma mark - BGM
// 添加背景音乐
/// - Parameters:
///   - bgmPath: 音乐素材的文件路径
///   - startAtVideoTime: BGM在时间轴中开始播放的时间(默认为0)，比如主轨道是一个10s视频，选择在第5s才开始添加BGM播放，则传5s对应的CMTime
///   - timeRange: 截取音乐素材的播放的区间(默认为整首),如一首歌时长为0-90s，设置timerange可以截取0-90任意一段的音频来使用
///  - retrun: bgm素材唯一ID
- (int)addBgm:(NSString *)bgmPath startAtVideoTime:(CMTime)startAtVideoTime timeRange:(CMTimeRange)timeRange;

/// 根据唯一ID删除背景音乐
/// - Parameter bgmId: 背景音乐添加时生成的ID
- (BOOL)removeBgm:(int)bgmId;
/// 删除所有bgm
- (void)removeAllBgm;

/// 调整Bgm在视频中开始的位置
/// - Parameters:
///   - bgmId: 背景音乐添加时生成的ID
///   - startAtVideoTime: BGM在时间轴中开始播放的时间，比如主轨道是一个10s视频，选择在第5s才开始添加BGM播放，则传5s对应的CMTime
- (void)updateBgm:(int)bgmId atVideoTime:(CMTime)startAtVideoTime;

/// 设置背景音乐播放区间
/// - Parameters:
///   - bgmId: 背景音乐添加时生成的ID
///   - timeRange: 截取音乐素材的播放的区间(默认为整首),如一首歌时长为0-90s，设置timerange可以截取0-90任意一段的音频来使用
- (void)updateBgm:(int)bgmId playRange:(CMTimeRange)timeRange;

/// 设置背景音乐播放音量
/// - Parameters:
///   - bgmId: 背景音乐添加时生成的ID
///   - volume: 音量 0-1f
- (void)setBgm:(int)bgmId volume:(float)volume;

/// 设置背景音乐的额外信息
/// - Parameters:
///   - bgmId: 背景音乐添加时生成的ID
///   - extraInfo: 自定义保存信息
- (void)setBgm:(int)bgmId extraInfo:(NSDictionary *)extraInfo;
- (NSDictionary *)getBgmExtraInfo:(int)bgmId;
/// 设置所有背景音乐播放音量
/// - Parameters:
///   - volume: 音量 0-1f
- (void)setAllBgmVolume:(float)volume;
- (float)getBgmVolume:(int)bgmId;

/// 调整背景音乐播放速度
/// - Parameters:
///   - bgmId: 背景音乐添加时生成的ID
///   - speed: 播放速度 1正常 大于1快 小于1慢
- (void)setBgm:(int)bgmId speed:(float)speed;
- (float)getBgmSpeed:(int)bgmId;

/// 获取当前使用的背景音乐
- (NSArray<TAVBgmInfo *> *)getBgmInfos;
#pragma mark - preview

/// 初始化预览播放器
/// - Parameter param: 需要给一个父容器和渲染方式
- (void)setPreview:(TAVPreviewParam *)param;

/// 设置循环播放
/// - Parameter isLoop: 是否循环播放
- (void)setPlayLoop:(BOOL)isLoop;
- (BOOL)isPlayLoop;

/// 播放
- (void)play;

/// 从某个时间开始播放
/// - Parameter time: 开始播放时间
- (void)playAtTime:(CMTime)time;

/// 暂停
- (void)pause;

/// 停止
- (void)stop;

/// 刷新预览播放器
- (void)refresh;

//- (void)replay;

/// seek到指定时间播放
/// - Parameter seekTime: 指定的播放时间
- (void)seekToTime:(CMTime)seekTime;

/// seek到指定时间播放
/// - Parameters:
///   - seekTime: 指定的播放时间
///   - completion: seek完成回调
- (void)seekToTime:(CMTime)seekTime completion:(void (^)(BOOL finished))completion;

/// 释放播放器
- (void)destoryPlayer;

/// 获取预览总时长（添加资源后需要调用flushImmediately后才能准确获取）
- (CMTime)getTotalDuration;

/// 主轨道总时长（添加资源后，可即时获取时长）
- (NSTimeInterval)duration;

/// 是否在播放中
- (BOOL)isPlaying;

/// 当前播放的时间
- (NSTimeInterval)currentTime;
- (CMTime)currentCMTime;

/// 设置预览填充模式
/// - Parameter fillMode: 填充模式
- (void)setPlayerViewFillMode:(TAVPlayerFillMode)fillMode;

/// 获取播放器填充模式
- (TAVPlayerFillMode)playerViewFillMode;

/// 注册播放状态监听者
/// - Parameter observer: 监听者
- (void)registerPlayerObserver:(id<TAVEditorPlayerObserver>)observer;

/// 注销播放状态监听者
/// - Parameter observer: 监听者
- (void)resignObserver:(id<TAVEditorPlayerObserver>)observer;

// 设置当前渲染大小
- (void)setRenderSize:(CGSize)renderSize;
/// 获取当前渲染大小
- (CGSize)getRenderSize;

//- (void)setPainting:(int)fillModel color:(NSString *)hexColor;

/// 获取当前播放器的完整avaseet
- (AVAsset *)currentAVAsset;
/// 获取当前播放器的完整videoComposition
- (AVVideoComposition *)currentComposition;

#pragma mark - filter
/// 设置滤镜
/// - Parameters:
///   - lutPath: 滤镜素材路径
///   - strength: 滤镜强度
///   - timeRange: 滤镜生效区间
/// - retrun: 创建滤镜时生成的唯一ID
- (int)addFilter:(NSString *)lutPath strength:(float)strength timeRange:(CMTimeRange)timeRange;

/// 调整滤镜强度与时间
/// - Parameters:
///   - lutId: 创建滤镜时生成的唯一ID
///   - strength: 滤镜强度
///   - timeRange: 滤镜生效区间
- (void)setFilter:(int)lutId Strength:(float)strength timeRange:(CMTimeRange)timeRange;

/// 删除滤镜
/// - Parameter lutId: 创建滤镜时生成的唯一ID
- (void)removeFilter:(int)lutId;

/// 删除所有滤镜效果
- (void)removeAllFilter;

/// 保存轨道下标(业务使用)
- (void)setFilter:(int)lutId trackIndex:(int)trackIndex;

/// 获取当前所有设置的滤镜效果
- (NSArray<TAVFilterInfo *> *)getFilterInfos;

#pragma mark - effect
/// 设置特效
/// - Parameters:
///   - path: 特效素材路径
///   - timeRange: 特效对应的时间区域
/// - return：特效id
- (int)addMotionEffect:(NSString *)path timeRange:(CMTimeRange)timeRange;

/// 更新特效播放时间
/// - Parameters:
///   - effectId: 创建特效时生成的id
///   - timeRange: 特效展示区间
- (void)updateMotionEffect:(int)effectId playRange:(CMTimeRange)timeRange;

/// 删除特效
/// - Parameter effectId: 创建特效时生成的id
- (void)removeMotionEffect:(int)effectId;

/// 删除所有特效
- (void)removeAllMotionEffect;

/// 保存轨道下标(业务使用)
- (void)setMotionEffect:(int)effectId trackIndex:(int)trackIndex;

/// 获取当前所有设置的特效
- (NSArray<TAVEffectInfo *> *)getEffectInfos;

#pragma mark - export video
/// 生成并导出视频
/// - Parameters:
///   - exportConfig: 导出视频配置,传nil使用默认配置(分辨率使用预览分辨率、帧数为30、码率为8*1000*1000)
///   - progressBlock: 导出视频进度回调
///   - completion: 导出视频完成回调
- (void)generateVideo:(TAVEditorGenerateConfig *)config 
           outputPath:(NSString *)outputPath
             progress:(void (^)(CGFloat progress))progressBlock
           completion:(void (^)(NSError *error, NSString *outputPath))completion;

/// 生成并导出视频
/// - Parameters:
///   - TAVVideoCompressed: 导出分辨率
///   - progressBlock: 导出视频进度回调
///   - completion: 导出视频完成回调
- (void)generateVideoWithCompressed:(TAVVideoCompressed)compressed
                         outputPath:(NSString *)outputPath
                           progress:(void (^)(CGFloat progress))progressBlock
                         completion:(void (^)(NSError *error, NSString *outputPath))completion;

/// 快速生成并导出视频
/// - Parameters:
///   - exportConfig: 导出视频配置,传nil使用默认配置(分辨率使用预览分辨率、帧数为30、码率为8*1000*1000)
///   - progressBlock: 导出视频进度回调
///   - completion: 导出视频完成回调
- (void)quickGenerateVideo:(TAVEditorGenerateConfig *)config
                outputPath:(NSString *)outputPath
                  progress:(void (^)(CGFloat progress))progressBlock
                completion:(void (^)(NSError *error, NSString *outputPath))completion;

/// 快速生成并导出视频
/// - Parameters:
///   - TAVVideoCompressed: 导出分辨率
///   - progressBlock: 导出视频进度回调
///   - completion: 导出视频完成回调
- (void)quickGenerateVideoWithCompressed:(TAVVideoCompressed)compressed
                              outputPath:(NSString *)outputPath
                                progress:(void (^)(CGFloat progress))progressBlock
                              completion:(void (^)(NSError *error, NSString *outputPath))completion;

/// 取消生成视频
/// - Parameter completion: 取消完成回调
- (void)cancelGenerate:(void(^ __nullable)(void))completion;

/// 暂停导出
- (void)pauseGenerate;

/// 继续导出
- (void)resumeGenerate;

#pragma mark - template
/// 获取视频模版坑位信息
/// - Parameter templatePath: 视频模版素材路径
- (TAVTemplateConfig * _Nullable)getTemplateConfig:(NSString * _Nullable)templatePath;

/// 应用模板
/// - Parameter templatePath: 视频模版素材路径
/// - return: 0成功，-1路径无效，-2无权限
- (TAVTemplateCode)setTemplate:(NSString * _Nullable)templatePath;

#pragma mark - pip
/// 获取画中画操作类
- (id<ITAVPipManager> _Nullable)getPipManager;

#pragma mark - sticker
/// 获取贴纸操作类
- (id<ITAVStickerManager> _Nullable)getStickerManager;

#pragma mark - video frame tool
/// 获取抽帧操作类
- (id<ITAVFrameProvider> _Nullable)getVideoFrameManager;

#pragma mark - draft
/// 获取草稿操作类
- (id<ITAVDraftManager> _Nullable)getDraftManager;

@end

