//
//  TAVEditorDefinitions.h
//  TAVMagic
//
//  Created by chavezchen on 2023/12/6.
//

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>
#import <TAVMagic/TAVMediaPlayer/TAVPlayerDefine.h>
#import <UIKit/UIKit.h>

typedef NS_ENUM(NSInteger, TAVEditorMediaType) {
    TAVEditorMediaTypeUnknown = 0,      // 未知
    TAVEditorMediaTypeVideo   = 1 << 0, // 视频
    TAVEditorMediaTypeAudio   = 1 << 1, // 音频
    TAVEditorMediaTypeImage   = 1 << 2, // 照片
};

typedef NS_ENUM(NSInteger, TAVTemplateCode) {
    TAVTemplateCodeSuccess = 0,         // 成功
    TAVTemplateCodePathInvalid   = 1 ,  // 模版路径无效
    TAVTemplateCodeAuthInvalid   = 2    // 无权限
};

typedef NS_ENUM(NSUInteger, TAVLogLevel) {
    TAVLogLevelUnknown = 0,
    TAVLogLevelDefault,
    TAVLogLevelVerbose,
    TAVLogLevelDebug,
    TAVLogLevelInfo,
    TAVLogLevelWarn,
    TAVLogLevelError,
    TAVLogLevelFatal
};
@protocol TAVLogListener <NSObject>
/// @brief 日志监听回调Block     Log monitoring callback
/// @param loggerLevel 返回当前日志等级     Return the current log level
/// @param logMessage 返回当前日志信息     Return the current log information
- (void)onLog:(TAVLogLevel)loggerLevel logMessage:(NSString * _Nonnull)logMessage;
@end

// 预览播放监听代理
@protocol TAVEditorPlayerObserver <NSObject>
@optional
- (void)mediaPlayerItemDidChange:(AVPlayerItem *)playerItem;
- (void)mediaPlayerProgressDidChange:(CMTime)currentTime;
- (void)mediaPlayerStateDidChange:(TAVPlaybackState)state;
- (void)mediaPlayerSpeedDidChange:(CGFloat)speed;
- (void)mediaPlayerDidPlayToEnd;
- (void)mediaPlayerVideoRectDidChange:(CGRect)videoRect;
@end

// 预览参数
@class TAVWidgetContentView;

@interface TAVPreviewParam : NSObject
@property (nonatomic, weak) UIView *videoView; // 播放器的父容器,必填
@property (nonatomic, assign) TAVPlayerFillMode fillModel; // 拉伸模式，选填
@property (nonatomic, assign) BOOL loop; // 循环播放，默认为NO,选填
@property (nonatomic, weak) id<TAVEditorPlayerObserver> observer; // 播放状态观察者，选填
/// 设置贴纸/画中画等挂件的操作容器视图,不设置则使用默认，自定义该容器可以绘制一些居中辅助线之类的。
@property (nonatomic, weak) TAVWidgetContentView *widgetContentView;
@end

#pragma mark - 导出配置
typedef NS_ENUM(NSInteger, TAVVideoCompressed) {
    TAVVideoCompressed480P   = 0 ,  // 压缩至480P分辨率（640*480）
    TAVVideoCompressed540P   = 1 ,  // 压缩至540P分辨率 (960*540)
    TAVVideoCompressed720P   = 2 ,  // 压缩至720P分辨率 (1280*720)
    TAVVideoCompressed1080P  = 3    // 压缩至1080P分辨率 (1920*1080)
};

@interface TAVGenerateVideoConfig : NSObject
@property (nonatomic, assign) int videoBitRate;     // 平均码率，默认8*1000*1000
@property (nonatomic, assign) int frameRate;        // 帧率，默认每秒30帧
@property (nonatomic, assign) CGSize resolution;    // 分辨率,不设置默认使用预览的分辨率
@property (nonatomic, assign) CMTimeRange exportTimeRange;  // 导出的时间区域，默认为全部时长
@end

@interface TAVGenerateAudioConfig : NSObject
@property (nonatomic,assign) int sampleRate;     // 采样率，默认44100
@property (nonatomic,assign) int audioBitRate;   // 编码码率，默认 128000
@property (nonatomic,assign) int channelCount;   // 声道数量，默认 2 双声道
@end

@interface TAVEditorGenerateConfig : NSObject
@property (nonatomic, strong) TAVGenerateVideoConfig *videoConfig;
@property (nonatomic, strong) TAVGenerateAudioConfig *audioConfig;
@end

#pragma mark - 模版功能
typedef NS_ENUM(NSInteger, TAVTemplateMediaType) {
    /** 视频 */
    TAVTemplateMediaTypeVideo   = 1,
    /** 照片 */
    TAVTemplateMediaTypeImage   = 2,
    /** 视频+照片 */
    TAVTemplateMediaTypeMulti   = 3,
};

static const int TEMPLATE_DEFAULT_BGM_ID = -10001; // 模版固定bgmId

// 模版视频坑位对象
@interface TAVClipPlaceHolder : NSObject
@property (nonatomic, assign) CMTime contentDuration; // 内容展示时间
@end

// 模版音频坑位对象
@interface TAVAudioPlaceHolder : NSObject
@property (nonatomic, copy) NSString *path; // bgm path
@property (nonatomic, copy) NSString *name;
@property (nonatomic, assign) NSInteger bgmId; 
@property (nonatomic, assign) CMTime duration; // 时长
@property (nonatomic, assign) CGFloat volume; // 音量 0-1
@property (nonatomic, assign) CMTime startOffset; // 在时间轴中的开始时间
@property (nonatomic, assign) NSInteger loopCount; // 0 表示不播放, 1 表示循环一次, -1 表示无限循环
@end

// 视频模版配置
@interface TAVTemplateConfig : NSObject
/// 1 仅能选视频, 2 仅能选照片, 3 可以选照片和视频
@property (nonatomic, assign) TAVTemplateMediaType mediaType;
/// 用户上传素材最少个数, 0 表示不限制
@property (nonatomic, assign) int minAssetCount;
/// 用户上传素材最多个数, 0 表示不限制
@property (nonatomic, assign) int maxAssetCount;
/// 视频坑位数组
@property (nonatomic, copy) NSArray<TAVClipPlaceHolder *> *clipPlaceHolders;
/// 音频坑位数组
@property (nonatomic, copy) NSArray<TAVAudioPlaceHolder *> *clipAudioHolders;
@end

#pragma mark -  资源片段信息
@class TAVMediaAssetCropInfo;
@interface TAVClipInfo : NSObject
// 唯一标识
@property (nonatomic, copy) NSString *uniqueID;
// 下标
@property (nonatomic, assign) NSInteger index;
/// 资源类型
@property (nonatomic, assign) TAVEditorMediaType mediaType;
/// 资源路径
@property (nonatomic, copy) NSString *filePath;
/// 视频资源对应的AVAsset资源
@property (nonatomic, strong) AVAsset *asset;
/// 图片
@property (nonatomic, strong) UIImage *image;
/// 变速
@property (nonatomic, assign) float speed;
/// 资源尺寸
@property (nonatomic, assign) CGSize naturalSize;
/// 资源在时间轴内的范围,受speed影响
@property (nonatomic, assign) CMTimeRange timeRange;
/// 在时间轴中的开始时间
@property (nonatomic, assign) CMTime startTime;
/// 资源实际的范围，比如10s的视频，使用了其中的3s-7s，这个就是[3,4]
@property (nonatomic, assign) CMTimeRange sourceUseRange;
/// 原始资源的时间区间,比如10s视频，一直都是[0,10]
@property (nonatomic, assign) CMTimeRange sourceRange;
/// 在时间轴中的总时长
@property (nonatomic, assign) CMTime duration;
/// 转场重叠时长
@property (nonatomic, assign) CMTime transitionIntersectionDuration;
/// 音量大小
@property (nonatomic, assign) CGFloat volume;
/// 裁剪形变
@property (nonatomic, assign) CGAffineTransform clipTransform;
/// 剪裁大小
@property (nonatomic, assign) CGSize clipSize;
/// 裁剪页的信息
@property (nonatomic, strong, nullable) TAVMediaAssetCropInfo *cropInfo;
@end

// 添加的bgm信息
@interface TAVBgmInfo : NSObject
/// 背景音乐id
@property (nonatomic, assign) int bgmId;
/// 背景音乐路径
@property (nonatomic, copy) NSString *bgmPath;
/// 背景音乐音量
@property (nonatomic, assign) float volume;
/// 背景音乐速率
@property (nonatomic, assign) float speed;
/// 背景音乐在时间轴的开始播放位置
@property (nonatomic, assign) CMTime startTime;
/// 背景音乐在时间轴的播放区间
@property (nonatomic, assign) CMTimeRange timeRange;
/// 背景音乐资源实际区间
@property (nonatomic, assign) CMTimeRange sourceRange;
/// 资源实际的范围，比如10s的视频，使用了其中的3s-7s，这个就是[3,4]
@property (nonatomic, assign) CMTimeRange sourceUseRange;
@property (nonatomic, strong) NSDictionary *extraInfo;

@end


@interface TAVEffectInfo : NSObject
/// 动效id
@property (nonatomic, assign) int effectId;
/// 动效路径
@property (nonatomic, copy) NSString *effectPath;
/// 动效在时间轴的播放区间
@property (nonatomic, assign) CMTimeRange timeRange;
/// 扩展字段
@property (nonatomic, strong) NSDictionary *extraInfo;
/// 轨道下标(可用于业务)
@property (nonatomic, assign) NSInteger trackIndex;
@end

@interface TAVFilterInfo : NSObject
/// 滤镜id
@property (nonatomic, assign) int lutId;
/// 滤镜路径
@property (nonatomic, copy) NSString *lutPath;
/// 滤镜强度
@property (nonatomic, assign) CGFloat strength;
/// 动效在时间轴的播放区间
@property (nonatomic, assign) CMTimeRange timeRange;
/// 扩展字段
@property (nonatomic, strong) NSDictionary *extraInfo;
/// 轨道下标(可用于业务)
@property (nonatomic, assign) NSInteger trackIndex;
@end
