//
//  TAVGeometryFunction.h
//  TAVStickerKit
//
//  Created by Victor Tian on 2018/11/2.
//  Copyright © 2018 Tencent Inc. All rights reserved.
//

#ifndef TAVGeometryFunction_h
#define TAVGeometryFunction_h

#import <AVFoundation/AVFoundation.h>
#import <UIKit/UIKit.h>

#pragma mark - CGPoint
NS_INLINE CGFloat TAVCGPointGetDistance(CGPoint first, CGPoint second) {
    CGFloat deltaX = second.x - first.x;
    CGFloat deltaY = second.y - first.y;
    return sqrt(deltaX*deltaX + deltaY*deltaY );
};

NS_INLINE CGPoint TAVCGPointAdd(CGPoint point1, CGPoint point2) {
    return CGPointMake(point1.x + point2.x, point1.y + point2.y);
}

NS_INLINE CGPoint TAVCGPointAddValue(CGPoint point, CGFloat value) {
    return CGPointMake(point.x + value, point.y + value);
}

NS_INLINE CGPoint TAVCGPointSubtract(CGPoint point1, CGPoint point2) {
    return CGPointMake(point1.x - point2.x, point1.y - point2.y);
}

NS_INLINE CGPoint TAVCGPointMakeScale(CGPoint point, CGFloat scale) {
    return CGPointMake(point.x * scale, point.y * scale);
}

NS_INLINE bool TAVCGPointIsInvalid(CGPoint point) {
    return isnan(point.x) || isnan(point.y) || isinf(point.x) || isinf(point.y);
}

NS_INLINE bool TAVCGPointIsValid(CGPoint point) {
    return !TAVCGPointIsInvalid(point);
}

#pragma mark - CGSize
NS_INLINE CGSize TAVCGSizeMakeScale(CGSize size, CGFloat scale) {
    return CGSizeMake(size.width * scale, size.height * scale);
}

NS_INLINE CGSize TAVCGSizeAddValue(CGSize size, CGFloat value) {
    return CGSizeMake(size.width + value, size.height + value);
}

NS_INLINE bool TAVCGSizeIsInvalid(CGSize size) {
    return isnan(size.width) || isnan(size.height) || isinf(size.width) || isinf(size.height);
}

NS_INLINE bool TAVCGSizeIsValid(CGSize size) {
    return !TAVCGSizeIsInvalid(size);
}

#pragma mark - CGRect
NS_INLINE CGPoint TAVCGRectGetCenter(CGRect rect) {
    return CGPointMake(CGRectGetMidX(rect), CGRectGetMidY(rect));
}

NS_INLINE CGRect TAVCGRectMakeScale(CGRect rect, CGFloat scale) {
    return (CGRect){rect.origin, CGSizeMake(rect.size.width * scale, rect.size.height * scale)};
}

NS_INLINE CGRect TAVCGRectMakeAllScale(CGRect rect, CGFloat scale) {
    return (CGRect){CGPointMake(rect.origin.x * scale, rect.origin.y * scale),
                    CGSizeMake(rect.size.width * scale, rect.size.height * scale)};
}

NS_INLINE bool TAVCGRectIsInvalid(CGRect rect) {
    return TAVCGPointIsInvalid(rect.origin) || TAVCGSizeIsInvalid(rect.size) || CGRectIsEmpty(rect) || CGRectIsInfinite(rect);
}

NS_INLINE bool TAVCGRectIsValid(CGRect rect) {
    return !TAVCGRectIsInvalid(rect);
}

NS_INLINE bool TAVCGRectApproxEqualToRect(CGRect rect1, CGRect rect2, CGFloat offset) {
    return ABS(rect1.origin.x - rect2.origin.x) <= offset && ABS(rect1.origin.y - rect2.origin.y) <= offset &&
    ABS(rect1.size.width - rect2.size.width) <= offset && ABS(rect1.size.height - rect2.size.height) <= offset;
}

NS_INLINE CGRect TAVCGRectConverToInt(CGRect rect) {
    return CGRectMake((int)rect.origin.x, (int)rect.origin.y, (int)rect.size.width, (int)rect.size.height);
}

NS_INLINE CGRect TAVCGRectRoundToInt(CGRect rect) {
    return CGRectMake(round(rect.origin.x), round(rect.origin.y), round(rect.size.width), round(rect.size.height));
}

NS_INLINE CGRect TAVCGRectContentInset(CGRect rect, UIEdgeInsets insets) {
    rect.origin.x       += insets.left;
    rect.origin.y       += insets.top;
    rect.size.width     -= (insets.left + insets.right);
    rect.size.height    -= (insets.top + insets.bottom);
    return rect;
}

NS_INLINE CGRect TAVCGRectReverseInset(CGRect rect, UIEdgeInsets insets) {
    rect.origin.x       -= insets.left;
    rect.origin.y       -= insets.top;
    rect.size.width     += (insets.left + insets.right);
    rect.size.height    += (insets.top + insets.bottom);
    return rect;
}

NS_INLINE CGRect TAVCGRectNormalizeByRect(CGRect rect, CGRect renderRect) {
    CGFloat scaledX = CGRectGetMinX(rect) / CGRectGetWidth(renderRect);
    CGFloat scaledY = CGRectGetMinY(rect) / CGRectGetHeight(renderRect);
    CGFloat scaledWidth = CGRectGetWidth(rect) / CGRectGetWidth(renderRect);
    CGFloat scaledHeight = CGRectGetHeight(rect) / CGRectGetHeight(renderRect);
    return CGRectMake(scaledX, scaledY, scaledWidth, scaledHeight);
}

NS_INLINE CGRect TAVCGRectNormalizeBySize(CGRect rect, CGSize renderSize) {
    CGFloat scaledX = CGRectGetMinX(rect) / renderSize.width;
    CGFloat scaledY = CGRectGetMinY(rect) / renderSize.height;
    CGFloat scaledWidth = CGRectGetWidth(rect) / renderSize.width;
    CGFloat scaledHeight = CGRectGetHeight(rect) / renderSize.height;
    return CGRectMake(scaledX, scaledY, scaledWidth, scaledHeight);
}

NS_INLINE CGRect TAVCGRectConvertAnchorToCenter(CGRect rect) {
    CGFloat centerX = CGRectGetMidX(rect);
    CGFloat centerY = CGRectGetMidY(rect);
    return CGRectMake(centerX, centerY, CGRectGetWidth(rect), CGRectGetHeight(rect));
}

NS_INLINE CGRect TAVCGRectConvertAnchorToTopLeft(CGRect rect) {
    CGFloat originX = CGRectGetMinX(rect) - CGRectGetWidth(rect) / 2;
    CGFloat originY = CGRectGetMidY(rect) - CGRectGetHeight(rect) / 2;
    return CGRectMake(originX, originY, CGRectGetWidth(rect), CGRectGetHeight(rect));
}

#pragma mark - CGAffineTransform
NS_INLINE CGFloat TAVCGAffineTransformGetRotation(CGAffineTransform t) {
    return atan2(t.b, t.a);
}

NS_INLINE CGFloat TAVCGAffineTransformGetScaleX(CGAffineTransform t) {
    return  sqrt(t.a * t.a + t.c * t.c);
}

NS_INLINE CGFloat TAVCGAffineTransformGetScaleY(CGAffineTransform t) {
    return sqrt(t.b * t.b + t.d * t.d);
}

NS_INLINE CGFloat TAVCGAffineTransformGetScale(CGAffineTransform t) {
    return MAX(TAVCGAffineTransformGetScaleX(t), TAVCGAffineTransformGetScaleY(t));
}

NS_INLINE CGPoint TAVCGAffineTransformGetTranslation(CGAffineTransform t) {
    return CGPointMake(t.tx, t.ty);
}

/// check transform had contact mirror
NS_INLINE BOOL TAVCGAffineTransformIsMirror(CGAffineTransform t) {
    return ABS(t.a + t.d) < FLT_EPSILON && ABS(t.b - t.c) < FLT_EPSILON;
}

#pragma mark - CMTime

/// Indicates whether a time is contained within a time range. Contains both boundary.
/// @param range A CMTimeRange to be tested for inclusion.
/// @param time A CMTime to be tested for inclusion.
NS_INLINE Boolean TAVTimeRangeContainsTime(CMTimeRange range, CMTime time) {
    return
    CMTimeCompare(time, range.start) >= 0 &&
    CMTimeCompare(time, CMTimeRangeGetEnd(range)) <= 0;
}

NS_INLINE Boolean TAVTimeRangeIsZero(CMTimeRange range) {
    return (range.start.value == 0 && range.duration.value == 0) || CMTimeRangeEqual(kCMTimeRangeZero, range);
}

NS_INLINE NSTimeInterval CMTimeGetMicroseconds(CMTime time) {
    return CMTimeGetSeconds(time) * 1000000.0;
}

#endif /* TAVGeometryFunction_h */
