//
//  TAVMediaBaseEffect.h
//  TAVCut
//
//  Created by Victor Tian on 2021/1/13.
//  Copyright © 2021 Tencent. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>
#import "TAVMediaLightAssetFragmentItem.h"
#import "TAVMediaEntityGroupArrangeConfig.h"
 
NS_ASSUME_NONNULL_BEGIN
typedef NS_ENUM(NSUInteger, TAVLightAssetState) {
    TAVLightAssetStateUnknown,
    TAVLightAssetStateEnable,
    TAVLightAssetStateDisable
};

typedef NS_ENUM(NSUInteger, TAVMediaEffectType) {
    /** 贴纸 */
    TAVMediaEffectTypeSticker,
    /** 特效 */
    TAVMediaEffectTypeEffect,
    /** 歌词 */
    TAVMediaEffectTypeLyric,
    /** 美颜 */
    TAVMediaEffectTypeBeauty,
    /** LUT滤镜 */
    TAVMediaEffectTypeLookup,
    /** 调节 */
    TAVMediaEffectTypeAdjustment,
    /** 水印 */
    TAVMediaEffectTypeWatermark,
    /** 转场 */
    TAVMediaEffectTypeTransition,
    /** 片头 */
    TAVMediaEffectTypeBeginning,
    /** 片尾 */
    TAVMediaEffectTypeEnding,
    /** 特殊漫画效果 */
    TAVMediaEffectTypeComics,
    /** 动效 */
    TAVMediaEffectTypeMotionEffect,
    /** 背景 */
    TAVMediaEffectTypeBackground,
    /** HDR */
    TAVMediaEffectTypeHDR,
    /** 模板 */
    TAVMediaEffectTypeTemplate,
    /** 预渲染 */
    TAVMediaEffectTypePrerender,
    /** 画中画 */
    TAVMediaEffectTypePictureInPicture,
};

typedef NS_ENUM(NSUInteger, TAVEffectSource) {
    /** 固定素材或者用户选择的素材 */
    TAVEffectSourceDefualt,
    /** 模板效果素材 */
    TAVEffectSourceTemplate,
    /** light模板效果素材 */
    TAVEffectSourceLightTemplate,
    /** 解析背景模版，根据其entity创建得到的effectItem，和entity有一一对应关系，一般来源于空间说说 */
    TAVEffectSourceLightMoodTemplate,
    /** 标记加载背景模版后，用户手动创建的item，和模版中的entity无关 */
    TAVEffectSourceMoodUser,
    /** 心情日记素材 */
    TAVEffectSourceZShowTemplate,
    /** 解析背景模版，根据模版中的特定背景文字entity，初始化创建得到的textItem，渲染逻辑同用户手动添加的item */
    TAVEffectSourceLightMoodText,
    /** 多层文字 layer 素材 */
    TAVEffectSourceMultiTextLayer,
    /** 马赛克涂鸦生成的贴纸，不能选中 */
    TAVEffectSourceMosaicGraffiti
};

@protocol TAVLightEffectItemProtocol;
@class TAVLAKNode,LAKEntity,LAKComponent,TAVLAKContext,LAKComponentBase;

@interface TAVMediaBaseEffect : NSObject <NSCopying, YYModel, YYModelForMd5,TAVLightEffectItemProtocol>

/**
 素材id，默认自动生成
 */
@property (nonatomic, copy) NSString *identifier;
/**
 效果的唯一id，内部自动生成，不可修改
 */
@property (nonatomic, copy, readonly) NSString *uniqueID;
/**
 素材分类id
 */
@property (nonatomic, copy, nullable) NSString *categoryID;
/**
 特效名称
 */
@property (nonatomic, copy, nullable) NSString *name;
/**
 特效文件路径
 */
@property (nonatomic, copy, nullable) NSString *filePath;
/**
 特效类型
 */
@property (nonatomic, assign) TAVMediaEffectType type;
/**
 特效来源
 */
@property (nonatomic, assign) TAVEffectSource source;
/**
 渲染顺序
*/
@property (nonatomic, assign) NSInteger layerIndex;
/**
 轨道索引，用于UI业务显示
*/
@property (nonatomic, assign) NSInteger trackIndex;
/**
 效果需要应用的时间区域
 */
@property (nonatomic, assign) CMTimeRange timeRange;

/**
 创建时间，since 1970的时间戳
 */
@property (nonatomic, assign) NSTimeInterval createTime;
/**
 扩展字段
 */
@property (nonatomic, strong) NSDictionary *extraInfo;

/// 该数据描述属于哪一个类别
@property (nonatomic, assign) TAVMediaGroup mediaGroup;
/**
 EffectSource发生变化，做相应操作
 */
- (void)updateEffectSource;

/**
 TAVLightEffectItemProtocol协议属性，是否使用LightSDK渲染特效，默认为NO，由业务方赋值
 */
@property (nonatomic, assign) BOOL useLightSDKEffect;

/**
 TAVMediaItem对应的LAK entity id
 */
@property (nonatomic, assign) NSInteger lightEntityId;

/**
 TAVMediaItem对应的lightTemplatePath
 */
@property (nonatomic, strong) NSString *lightTemplatePath;

/**
 是否是来自背景模版中的背景贴纸
 */
@property (nonatomic, assign) BOOL isMoodBackground;

/**
 是否是来自背景模版中的文字贴纸
 */
@property (nonatomic, assign) BOOL isMoodText;

/**
 背景模版需要的说说内容
 */
@property (nonatomic, strong, nullable) NSString *moodWords;

/// 描述子素材模式是否生效，若未Enable表示当前消费的素材为子素材
/// 考虑到多贴纸场景，业务提前判断赋值，减少CreateEntity时性能损耗
@property (nonatomic, assign) TAVLightAssetState lightAssetFragmentState;

/**
 业务方主动调用，将mediaItem对应的entity从context中去除
 */
- (void)removeFromLAKContext;

/**
 用以存储当前特效加入到的LAK环境，由父类持有，子类如果需要序列化，需要将此属性列入黑名单
 */
@property (nonatomic, strong) NSMutableArray <TAVLAKNode *>*nodeArray;

/**
 TAVLightEffectItemProtocol协议方法，由具体子类实现，具体子类实现的时候，调用一下super，以便赋值LAKContext
 */
- (LAKEntity *)createEntityWithLAKContext:(TAVLAKContext *)context needResetLightAsset:(BOOL *)needResetLightAsset;
/**
 协议方法，如果子类成功将entity添加进渲染树，需要调用父类的此方法
 */
- (void)recordEffectItemForContext:(TAVLAKContext *)context
                        withEntity:(LAKEntity *)entity
                     withComponent:(LAKComponentBase *)component;
/**
 协议方法，寻找当前LAK环境下被加入到渲染树的Node
 */
- (TAVLAKNode *)checkoutCurrentNodeWithContext:(TAVLAKContext *)context;
/**
 协议方法，当前特效是否已经加入到传入的Context中，由父类实现，子类不用关心
 */
- (BOOL)contextContainEffectItemWithContext:(TAVLAKContext *)context;

/// 通过compontID查找当前LAK环境下载渲染树中的Node中的Component
/// @param componentID id
- (nullable LAKComponent *)currentComponentWithId:(NSInteger)componentID;

/**
 TAVMediaItem对应的子素材-用以描述子素材的相关信息,将当前信息中的templatePath，timeRange进行组合，获取
 */
- (TAVMediaLightAssetFragmentItem *)lightAssetFragmentItem;
/**
 当前Item所需要的Light的bundle资源
 */
- (NSArray<NSString *> *)needAILightBundles;
/**
 是否需要使用子素材模式
 */
- (BOOL)shouldUsingLightAssetFragment;

/// 获取当前Node对应的Entity中可编辑的Entity
/// @param context 上下文
- (LAKEntity *)editorEntityCurrentNodeWithContext:(TAVLAKContext *)context;

@end

NS_ASSUME_NONNULL_END
