//
//  TAVMediaPlayer.h
//  TAVStickerKit_Example
//
//  Created by Victor Tian on 2019/6/27.
//  Copyright © 2019 tencent. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>
#import "TAVPlayerDefine.h"

NS_ASSUME_NONNULL_BEGIN

void runOnMediaMainQueue(void (^block)(void));

typedef void (^TAVCompletionHandler)(BOOL finished);

extern NSErrorDomain const TAVMediaPlayerErrorDomain;

@class TAVMediaPlayer;
@protocol TAVMediaPlayerObserver <NSObject>

@optional
/**
 Tells the observer that the AVPlayer has been reset.

 @param mediaPlayer The TAVMediaPlayer object.
 @param player The TAVPlayer object.
 */
- (void)mediaPlayer:(TAVMediaPlayer *)mediaPlayer
     playerDidReset:(AVPlayer *)player;
/**
 Tells the observer that the current item has changed.
 
 @param mediaPlayer The TAVMediaPlayer object.
 @param playerItem The AVPlayerItem object.
 */
- (void)mediaPlayer:(TAVMediaPlayer *)mediaPlayer
      itemDidChange:(AVPlayerItem *)playerItem;
/**
 Tells the observer that the current item has been replace set to AVPlayer.
 
 @param mediaPlayer The TAVMediaPlayer object.
 @param playerItem The AVPlayerItem object.
 */
- (void)mediaPlayer:(TAVMediaPlayer *)mediaPlayer
    itemDidReplaced:(AVPlayerItem *)playerItem;
/**
 Tells the observer that the state of TAVMediaPlayer has changed.
 
 @param mediaPlayer The TAVMediaPlayer object.
 @param state The TAVPlaybackState enumerate.
 */
- (void)mediaPlayer:(TAVMediaPlayer *)mediaPlayer
     stateDidChange:(TAVPlaybackState)state;
/**
 Tells the observer that the current time of TAVMediaPlayer has changed.
 
 @param mediaPlayer The TAVMediaPlayer object.
 @param currentTime A CMTime.
 */
- (void)mediaPlayer:(TAVMediaPlayer *)mediaPlayer
  progressDidChange:(CMTime)currentTime;
/**
 Tells the observer that the speed of TAVMediaPlayer has changed.
 
 @param mediaPlayer The TAVMediaPlayer object.
 @param speed The speed of the TAVMediaPlayer.
 */
- (void)mediaPlayer:(TAVMediaPlayer *)mediaPlayer
     speedDidChange:(CGFloat)speed;
/**
 Tells the observer that the current item has played to its end time.
 
 @param mediaPlayer The TAVMediaPlayer object.
 */
- (void)mediaPlayerDidPlayToEnd:(TAVMediaPlayer *)mediaPlayer;

/**
 Tells the observer that the player has refreshed.
 
 @param mediaPlayer The TAVMediaPlayer object.
 */
- (void)mediaPlayerDidRefresh:(TAVMediaPlayer *)mediaPlayer;


@end

@interface TAVMediaPlayer : NSObject

@property (nonatomic, strong, readonly) AVPlayer *player;
@property (nonatomic, strong, readonly) AVPlayerItem *currentItem;
@property (nonatomic, strong, readonly) AVPlayerItem *nextItem;

@property (nonatomic, strong, readonly) NSError *error;
@property (nonatomic, assign, readonly) TAVPlaybackState currentState;
@property (nonatomic, assign, readonly, getter=isPlaying) BOOL playing;
@property (nonatomic, assign, readonly) CMTime currentTime;
@property (nonatomic, assign, readonly) CMTime duration;

@property (nonatomic, assign) CGFloat speed;
@property (nonatomic, assign) CGFloat volume;
@property (nonatomic, assign) CMTimeRange validRange;
@property (nonatomic, assign) CMTime progressInterval;
@property (nonatomic, assign) BOOL shouldRepeat;
@property (nonatomic, assign) BOOL shouldResume;

/**
 Define the behavior of refresh.
 @discussion Refresh by seeking if NO. Default is NO.
 */
@property (nonatomic, assign, class, getter=isAdvancedRefreshEnabled) BOOL advancedRefreshEnabled;

/**
 Auto player auto pause when the application state is not activate. Deafults is NO.
 */
@property (nonatomic, assign) BOOL autoPauseWhenNotActivate;

/**
 Creates a new player to play the specified player item.

 @param playerItem The player item to play.
 @return An instance of TAVMediaPlayer.
 */
+ (instancetype)playerWithPlayerItem:(nullable AVPlayerItem *)playerItem;
/**
 Creates a new player to play the specified player item.
 
 @param playerItem The player item to play.
 @return An instance of TAVMediaPlayer.
 */
- (instancetype)initWithPlayerItem:(nullable AVPlayerItem *)playerItem;

/**
 Reset the current player item.

 @param playerItem The player item to play.
 */
- (void)resetPlayerItem:(nullable AVPlayerItem *)playerItem;

/**
 Reset the current player item without delay.

 @param playerItem The player item to play.
 @param needDelay if it takes effect immediately.
 */
- (void)resetPlayerItem:(AVPlayerItem *)playerItem withDelay:(BOOL)needDelay;

- (void)seekToTime:(CMTime)seekTime;
- (void)seekToTime:(CMTime)seekTime
        completion:(nullable TAVCompletionHandler)completion;

- (void)registerObserver:(id<TAVMediaPlayerObserver>)observer;
- (void)unregisterObserver:(id<TAVMediaPlayerObserver>)observer;

- (void)play;
- (void)playAtTime:(CMTime)time;
- (void)pause;
- (void)replay;
- (void)stop;
- (void)destory;
- (void)refresh;
- (void)refreshWithForced:(BOOL)forced;
/**
 If shouldResume is YES, play the current item. otherwise do nothing.
*/
- (void)resume;
/**
 When the player is paused, user can use this function to record a avplayerItem which will be used
 when the player be called useNextPlayerItemWithPlay or play.This function usually can be uesd
 to solve imprecise seek, like video cut etc.
 Finally if user call this function,the property nextItem will be assigned And will be assigned when user call
 useNextPlayerItemWithPlay or play
 @param playerItem will be assigned to nextItem
 */
- (void)replacePlayerItemWhenReplay:(AVPlayerItem *)playerItem;
/**
 If property nextItem isn't nil, user can use this function replace currentItem by nextItem immediately.
 @param needPlay if player play immediately
 */
- (void)useNextPlayerItemWithPlay:(BOOL)needPlay;
@end

NS_ASSUME_NONNULL_END
