//
//  TAVSticker.h
//  TAVStickerKit
//
//  Created by Victor Tian on 2019/1/20.
//  Copyright © 2019 Victor Tian. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>
#import <libpag/PAGFile.h>

NS_ASSUME_NONNULL_BEGIN

@class TAVStickerTextItem,TAVStickerImageItem,TAVStickerLayerInfo,TAVStickerAudioItem,TAVStickerSolidItem;

typedef CGFloat TAVStickerIndex NS_TYPED_EXTENSIBLE_ENUM;

UIKIT_EXTERN const TAVStickerIndex TAVStickerIndexMin;
UIKIT_EXTERN const TAVStickerIndex TAVStickerIndexMiddle;
UIKIT_EXTERN const TAVStickerIndex TAVStickerIndexMax;

typedef NS_ENUM(NSUInteger, TAVStickerMode) {
    TAVStickerModeDefault,
    TAVStickerModeActive,
    TAVStickerModeResign,
};

typedef NS_ENUM(NSUInteger, TAVStickerScaleMode) {
    TAVStickerScaleModeDefault,
    TAVStickerScaleModeAspectFill,
    TAVStickerScaleModeStretch
};

typedef NS_ENUM(NSUInteger, TAVStickerAnimationMode) {
    /** 循环 */
    TAVStickerAnimationModeDefault,
    /** 播放到最后一帧后静止 */
    TAVStickerAnimationModeFreeze,
    /** 当用户设定的时长大于特效本身设计时长时，慢速播放 */
    TAVStickerAnimationModeScaleUp,
    /** 适应特效时长，调整播放速度 */
    TAVStickerAnimationModeScaleFill,
    /** 使用PAG内置播放逻辑 */
    TAVStickerAnimationModePAGBuildIn
};

typedef NS_ENUM(NSUInteger, TAVStickerContentMode) {
    /** 响应区域：整个贴纸 */
    TAVStickerContentModeDefault,
    /** 响应区域：文本框 */
    TAVStickerContentModeText
};

typedef NS_ENUM(NSUInteger, TAVStickerSource) {
    /** 固定素材或者用户选择的素材中的贴纸 */
    TAVStickerSourceDefualt,
    /** 根据light模板素材中的配置节点生成的贴纸 */
    TAVStickerSourceLightTemplate,
    /** 马赛克涂鸦生成的贴纸，不能选中 */
    TAVStickerSourceMosaicGraffiti,
};

@class TAVSticker;
@protocol TAVStickerRenderer <NSObject>

@optional

/// Tells the renderer that the text data did change.
/// @param sticker TAVSticker object.
- (void)textDataDidChange:(TAVSticker *)sticker;
/// Tells the renderer that the image data did change.
/// @param sticker TAVSticker object.
- (void)imageDataDidChange:(TAVSticker *)sticker;
/// Tells the renderer that the solid color did change.
/// @param sticker TAVSticker object.
/// @param shouldReload Should reload data or not.
- (void)solidColorDidChange:(TAVSticker *)sticker
               shouldReload:(BOOL)shouldReload;
@end

@protocol TAVStickerProgress <NSObject>

@optional
/**
 provid progress for current sticker when render time changed,
 if you implement this method, the value in speed property will be ignored.

 @param sticker self
 @param renderTime current render time
 @return current progress
 */
- (CGFloat)progressForSticker:(TAVSticker *)sticker
                 atRenderTime:(CMTime)renderTime;

@end

@interface TAVSticker : NSObject<NSCopying>

/**
 Initializes and returns a sticker object.

 @param filePath The local path of the specified pag animation.
 @return Returns an initialized TAVSticker object.
 */
- (instancetype)initWithFilePath:(NSString *)filePath;

@property (nonatomic, copy) NSString *uniqueID;
@property (nonatomic, copy, readonly) NSString *filePath;
@property (nonatomic, strong, readonly) PAGFile *pagFile;
@property (nonatomic, assign, readonly) CGSize renderSize;
@property (nonatomic, assign, readonly) CMTime sourceDuration;
@property (nonatomic, strong, readonly) NSArray<TAVStickerTextItem *> *textList;
@property (nonatomic, strong, readonly) NSArray<TAVStickerImageItem *> *imageList;
@property (nonatomic, strong, readonly) NSArray<TAVStickerLayerInfo *> *infoList;
@property (nonatomic, strong, readonly) NSArray<TAVStickerAudioItem *> *audioList;
@property (nonatomic, strong, readonly) NSArray<TAVStickerSolidItem *> *solidList;
/// The number of video compositions.
@property (nonatomic, assign, readonly) NSInteger videoCount;

@property (nonatomic, assign) TAVStickerMode mode;
@property (nonatomic, assign) TAVStickerSource stickerSource;
@property (nonatomic, assign) NSInteger layerIndex;
@property (nonatomic, assign) CMTimeRange timeRange;
/// rotation & mirror transform.
/// without mirror t' = [ cos(angle) sin(angle) -sin(angle) cos(angle) 0 0 ]
/// with mirror t' = [ -cos(angle) -sin(angle) -sin(angle) cos(angle) 0 0 ]
@property (nonatomic, assign) CGAffineTransform transform;
@property (nonatomic, weak) id<TAVStickerProgress> progress;
@property (nonatomic, assign) BOOL shouldRepeat;
@property (nonatomic, assign) CGFloat speed;

@property (nonatomic, assign) CGFloat scale;
/// 旋转角度
@property (nonatomic, assign) CGFloat rotation;
/// 镜像属性
@property (nonatomic, assign, getter = isMirrored) BOOL mirrored;
/** {centerX, centerY, width, height} */
@property (nonatomic, assign) CGRect scaleRect;
@property (nonatomic, assign) CGFloat sizeRatio;
@property (nonatomic, assign) UIEdgeInsets contentInset;
@property (nonatomic, assign) TAVStickerScaleMode scaleMode;
@property (nonatomic, assign) TAVStickerContentMode contentMode;
/// 是否只含有文字layer
@property (nonatomic, readonly) BOOL isOnlyTextLayers;

/// 对于TAVMediaEffectItem中的sticker，内部画布大小可能和传入的不一样，默认NO
@property (nonatomic, assign) BOOL canManuallyUpdateRenderSize;
@property (nonatomic, assign) CGSize realRenderSize;

/// 源图片更新pag图层PAGImage的scaleMode
@property (nonatomic, assign) PAGScaleMode pagScaleMode;

/// 对于TAVMediaEffectItem中的sticker，现在内部强制设为TAVStickerScaleModeAspectFill。增加改属性，为YES时，外部可以改变
@property (nonatomic, assign) BOOL canManuallyUpdateScaleMode;

/// 用于控制特效动画的播放模式，目前支持循环、动画结束后静止和慢速三种模式，详见上面👆TAVStickerAnimationMode的注释。当shouldRepeat属性设置为YES时，此属性会同步被设为TAVStickerAnimationModeDefault；当shouldRepeat属性设置为NO时，此属性则会同步被设为TAVStickerAnimationModeFreeze。相对地，当此属性被设置为TAVStickerAnimationModeDefault时，shouldRepeat属性会被同步设置为YES，当此属性被设置为TAVStickerAnimationModeFreeze或TAVStickerAnimationModeScaleUp时，shouldRepeat属性会被同步设置为NO。当此属性为TAVStickerAnimationModeScaleUp时，特效动画播放的速度会根据用户设定的特效时长而拉长，因此speed属性设置的动画播放速度会失效。默认值为TAVStickerAnimationModeDefault（循环）。
@property (nonatomic, assign) TAVStickerAnimationMode animationMode;

/// 隐藏上屏态 defaults to NO
@property (nonatomic, assign) BOOL hideActiveMode;

- (BOOL)checkFile;
/// Copy the current PAGFile and bind the copied PAGFile to the TAVSticker,
/// and then auto update the PAGFile when the TAVSticker has been modified.
- (PAGFile *)copyPAGFile;
/// Bind the specified PAGFile to the current TAVSticker as main PAGFile
/// The textList of the sticker will be initialized to the ones of the specified PAGFile
/// and then auto update the PAGFile when the TAVSticker has been modified.
/// @param pagFile PAGFile object that should be bound.
- (void)bindPAGFile:(PAGFile *)pagFile;
/// Unbind the specified PAGFile.
/// @param pagFile PAGFile object that should be unbound.
- (void)unbindPAGFile:(PAGFile *)pagFile;

- (void)updateTextData;
- (void)updateImageData;

/// 更新solidlayer颜色
/// @param reload updateTextData会执行reload操作，所以这里如果只更新solidColor时，reload为YES，同时更新textdata，reload为NO，降低耗时
- (void)updateSolidColorDataWithReload:(BOOL)reload;

- (void)registerRenderer:(NSObject<TAVStickerRenderer> *)renderer;
- (void)unregisterRenderer:(NSObject<TAVStickerRenderer> *)renderer;

/**
 Calculate the sticker frame based on the render rect.

 @param renderRect The rect of the content view.
 @return Current sticker frame.
 */
- (CGRect)stickerFrameInRenderRect:(CGRect)renderRect;
/**
 Calculate the first textItem frame based on the render rect.

 @param renderRect The rect of the content view.
 @return First textItem frame.
 */
- (CGRect)textFrameInRenderRect:(CGRect)renderRect;
/**
 Calculate the first textItem normalized frame based on the self render size.

 @return First textItem normalized frame.
 */
- (CGRect)normalizedTextFrameInRenderSize;
/**
 Calculate the content rect based on the render rect.
 
 @param renderRect The rect of the content view.
 @return Current content rect.
 */
- (CGRect)contentFrameInRenderRect:(CGRect)renderRect;
/**
 Update current scale rect based on the frame of sticker.

 @param frame The frame of sticker.
 @param renderRect The rect of the content area.
 */
- (void)updateFrame:(CGRect)frame inRenderRect:(CGRect)renderRect;
/**
 Update current scale rect based on the frame of sticker.

 @param frame The frame of sticker.
 @param renderSize The size of the content area.
 */
- (void)updateFrame:(CGRect)frame inRenderSize:(CGSize)renderSize;

@end

@interface TAVSticker (TAVAnimationProgress)

/**
 Calculate the animation progress by current render time.

 @param renderTime Current render time.
 @return The animation progress.
 */
- (CGFloat)progressWithRenderTime:(CMTime)renderTime;

@end

NS_ASSUME_NONNULL_END
