//
//  TAVStickerView.h
//  TAVStickerKit
//
//  Created by Victor Tian on 2018/7/18.
//  Copyright © 2018 Tencent Inc. All rights reserved.
//

#import "TAVEditableView.h"
#import "TAVSticker.h"


typedef NS_ENUM(NSUInteger, TAVStickerTimelineMode) {
    /** In this mode you should control the animation by seekToTime: and updateProgress: methods. */
    TAVStickerTimelineModeDefault,
    /** In this mode you should control the animation by play, pause and replay mehtods. */
    TAVStickerTimelineModeAutoLoop,
};

typedef NS_ENUM(NSUInteger, TAVStickerAlignmentMode) {
    /** In this mode stickerView center limit to superview all around. */
    TAVStickerAlignmentModeEdgeCenter,
    /** In this mode stickerView edgeInset limit to superview all around. */
    TAVStickerAlignmentModeEdgeInset,
    /** In this mode, only a minimum visible line at the bottom must remain visible;the sticker can be moved out (hidden) in other directions. */
    TAVStickerAlignmentModeMinVisibleLine
};

@class TAVStickerView,TAVEditableView,TAVMediaStickerItem;
@protocol TAVEditableViewDelegate ;
@protocol TAVStickerViewDelegate <NSObject, TAVEditableViewDelegate>

@optional
/**
 Tells the delegate that the sticker has played to its end time.

 @param stickerView The sticker-view object that has play to end.
 */
- (void)stickerViewDidReachToEnd:(TAVStickerView *)stickerView;
/**
 Ask the delegate that the sticker should be hidden or not.

 @param stickerView The sticker-view object.
 @param currentTime The current time.
 */
- (BOOL)stickerView:(TAVStickerView *)stickerView
   shouldHideAtTime:(CMTime)currentTime;

- (void)stickerViewStartPanning:(TAVStickerView *)stickerView;
- (void)stickerViewChangePanning:(TAVStickerView *)stickerView;
- (void)stickerViewEndPanning:(TAVStickerView *)stickerView;
- (void)stickerViewOnDelete:(TAVStickerView *)stickerView;
- (void)stickerViewOnClicked:(TAVStickerView *)stickerView;
@end

@class TAVStickerAnimationView, TAVStickerLayerItem;
@interface TAVStickerView : TAVEditableView {
    TAVSticker *_sticker;
}
@property (nonatomic, weak) TAVMediaStickerItem *stickerItem;
@property (nonatomic, weak) AVPlayerItem *playerItem;
@property (nonatomic, strong, readonly) TAVSticker *sticker;
@property (nonatomic, weak) id<TAVStickerViewDelegate> delegate;
@property (nonatomic, strong, readonly) UIView *borderView;
@property (nonatomic, strong, readonly) UIView *contentView;
@property (nonatomic, strong, readonly) TAVStickerAnimationView *animationView;
@property (nonatomic, assign, readonly) TAVStickerMode mode;
@property (nonatomic, assign) TAVStickerTimelineMode timelineMode;
@property (nonatomic, assign, getter = isMirrored) BOOL mirrored;
@property (nonatomic, assign, readonly) CGSize contentSize;
@property (nonatomic, assign, readonly) CGFloat progress;
@property (nonatomic, assign, readonly) BOOL isPlaying;
@property (nonatomic, assign) NSInteger repeatCount;
@property (nonatomic, strong) UIColor *borderColor;
@property (nonatomic, assign) CGFloat borderWidth;
@property (nonatomic, assign) BOOL shouldRepeat;
@property (nonatomic, assign) TAVStickerAlignmentMode alignmentMode;
/// stickerView edgeInset limit to superview, only works in alignmentMode = TAVStickerAlignmentModeEdgeInset
@property (nonatomic, assign) UIEdgeInsets alignmentEdgeInset;
@property (nonatomic, assign) BOOL disableGestureWhenInactive; // 非活跃状态下禁止手势,默认为YES


/**
 Initializes and returns a sticker-view object having the given sticker.

 @param sticker The sticker object that should be loaded.
 @return Returns an initialized TAVStickerView object.
 */
- (instancetype)initWithSticker:(TAVSticker *)sticker;
- (instancetype)initWithStickerItem:(TAVMediaStickerItem *)stickerItem;

/**
 Flush the sticker animation.
 */
- (void)reloadData;
/**
 Reload sticker-view with the specified sticker.

 @param sticker The sticker object that should be loaded.
 */
- (void)reloadSticker:(TAVSticker *)sticker NS_REQUIRES_SUPER;
/**
 Update animation to the specified time.

 @param seekTime The animation time that want to seek.
 */
- (void)seekToTime:(CMTime)seekTime;
/**
 Update animation to the specified progress.

@param progress The animation time that want to seek.
*/
- (void)updateProgress:(CGFloat)progress;
/**
 Update the scale rect.
 */
- (void)updateScaleRect NS_REQUIRES_SUPER;

/**
 Active current sticker.
 */
- (void)becomeActive;
/**
 Resign current sticker.
 */
- (void)resignActive;
/**
 Switch the mode of current sticker.
 */
- (void)switchMode:(TAVStickerMode)mode;

/**
 Play the sticker animation.
 */
- (void)play;
/**
 Pause the sticker animation.
 */
- (void)pause;
/**
 Replay the sticker animation.
 */
- (void)replay;

/**
 * Returns all layers in the layer hierarchy that contains a specified point.
 */
- (NSArray<TAVStickerLayerItem *> *)layersUnderPoint:(CGPoint)touchPoint;

@end
