//
//  TEPanelView.m
//  TEBeautyKit
//
//  Created by tao yue on 2024/1/9.
//

#import <objc/runtime.h>
#import "TEPanelView.h"
#import "TECollectionViewCell.h"
#import "TEPanelDataProvider.h"
#import <Masonry/Masonry.h>
#import "TEUIProperty.h"
#import "TESlider.h"
#import "TEDownloader.h"
#import "TESwitch.h"
#import <MobileCoreServices/MobileCoreServices.h>
#import "TEUtils.h"
#import "TEToast.h"
#import <XMagic/XmagicConstant.h>
#import "TEUIConfig.h"
#import "TEBeautyProcess.h"
#import "TETabCollectionViewCell.h"
#import "TESliderValueCalculator.h"

#define beautyCollectionItemWidth 62
#define beautyCollectionItemHeight 90

static NSString *const kTECollectionViewCell = @"TECollectionViewCell";
static NSString *const kTETabCollectionViewCell = @"TETabCollectionViewCell";
static CGFloat const kPanelViewHeight = 230;
static CGFloat const kTitleCollectionHeight = 36;
static CGFloat const kCategoryTitleCollectionSpace = 10;
static CGFloat const kCategoryTitleCollectionItemWidth = 60;

// TEPanelView的类扩展，声明私有属性和协议遵循
@interface TEPanelView() <
UICollectionViewDelegate,               // 集合视图代理（处理交互事件）
UICollectionViewDataSource,             // 集合视图数据源（提供内容）
UICollectionViewDelegateFlowLayout,
UINavigationControllerDelegate,
UIImagePickerControllerDelegate,        // 图片选择控制器代理
TEBeautyProcessDelegate,
UIGestureRecognizerDelegate>             // 手势识别代理

/* 视图相关属性 */
@property (nonatomic, strong) UICollectionView *beautyCollection;           /// 美颜参数集合视图
@property (nonatomic, strong) UICollectionViewFlowLayout *beautyLayout;     /// 布局
@property (nonatomic, strong) TEBeautyProcess *beautyProcess;               /// 负责业务处理
@property (nonatomic, strong) UIButton *compareButton;                      /// 效果对比按钮
@property (nonatomic, strong) TESlider *teSlider;                           /// 定制滑动条组件
@property (nonatomic, strong) UIView *makeupOrLut;                          /// 美妆/LUT切换容器
@property (nonatomic, strong) UIView *blackView;                            /// 黑色遮罩视图
@property (nonatomic, strong) UIView *rightResetView;                       /// 右侧重置按钮容器
@property (nonatomic, strong) UICollectionView *titleCollection;            /// 一级标题
@property (nonatomic, strong) UICollectionViewFlowLayout *titleLayout;      /// 一级标题布局
@property (nonatomic, strong) UICollectionView *categoryCollection;         /// 二级标题
@property (nonatomic, strong) UICollectionViewFlowLayout *categoryLayout;   /// 二级标题布局
@property (nonatomic, strong) UILabel *beautyTitleLabel;                    /// 美颜标题标签
@property (nonatomic, strong) UIButton *backButton;                         /// 返回按钮
@property (nonatomic, strong) UILabel *processLabel;                        /// 进度提示标签
@property (nonatomic, strong) UIActivityIndicatorView *loadingView;         /// 加载指示器
/* 数据模型相关 */
@property (nonatomic, strong) TEPanelDataProvider *tePanelDataProvider; // 数据提供器
@property (nonatomic, strong) NSMutableArray<TEUIProperty *> *currentUIPropertyList; // 当前显示属性列表
@property (nonatomic, strong) TEUIProperty *currentRootUIProperty;      // 当前根UI
@property (nonatomic, strong) TEUIProperty *currentUIProperty;          // 当前选中UI属性项（tab对应的UIProperty）
@property (nonatomic, strong) TEUIProperty *currentCategoryUIProperty;  // 当前选中二级UI属性项
@property (nonatomic, strong) TEUIProperty *curProperty;                // 当前属性项
@property (nonatomic, strong) TEUIProperty *greenBackgroundV2Property;  // 绿幕v2

/* 状态标记 */
@property (nonatomic, assign) BOOL faceSwitchStatus;                   // 人脸检测开关状态
@property (nonatomic, assign) BOOL gestureSwitchStatus;                // 手势识别开关状态
@property (nonatomic, assign) BOOL isShowLoading;                      // 是否显示加载指示
@property (nonatomic, assign) BOOL showCompareBtn;                     // 是否显示对比按钮
@property (nonatomic, assign) BOOL enhancedMode;                       // 增强模式开关
@property (nonatomic, assign) BOOL showOrigin;                         // 是否显示原图对比
/* 功能参数 */
@property (nonatomic, assign) NSInteger makeupType;                    // 美妆类型
@property (nonatomic, assign) NSInteger showProgress;                  // 显示进度值
/* 其他 */
@property (nonatomic, strong) UITapGestureRecognizer *selfTapGesture;   // 自身点击手势

@end


@implementation TEPanelView

- (instancetype)initWithFrame:(CGRect)frame {
    if (self = [super initWithFrame:frame]) {
        self.showCompareBtn = YES;
        [self initData];
        [self initUI];
    }
    return self;
}

- (void)initData {
    self.tePanelDataProvider = [TEPanelDataProvider shareInstance];
    self.currentUIPropertyList = [self.tePanelDataProvider getAllPanelData];
    
    for (TEUIProperty *property in self.currentUIPropertyList) {
        [self getSelectedPoroerty:property.propertyList];
        for (TEUIProperty *subProperty in property.propertyList) {
            [self getSelectedPoroerty:subProperty.propertyList];
        }
    }
    
    self.currentUIProperty = [self getSelectedPoroerty:self.currentUIPropertyList];
    self.currentRootUIProperty = self.currentUIProperty;
    if (self.currentUIProperty.hasSubTitle) {
        self.currentCategoryUIProperty = [self getSelectedPoroerty:self.currentUIProperty.propertyList];
    }
}

- (void)initUI {
    self.backgroundColor = [UIColor clearColor];
    /// 添加对比视图
    [self addSubview:self.compareButton];
    [self.compareButton mas_makeConstraints:^(MASConstraintMaker *make) {
        make.right.offset(-5);
        make.width.height.mas_equalTo(35);
        make.top.offset(10);
    }];
    /// 添加滑杆
    [self addSubview:self.teSlider];
    [self.teSlider mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.offset(10);
        make.right.equalTo(self.compareButton.mas_left).offset(-5);
        make.centerY.equalTo(self.compareButton);
    }];
    self.teSlider.hidden = YES;
    /// 添加美颜滤镜切换视图
    [self addSubview:self.makeupOrLut];
    [self.makeupOrLut mas_makeConstraints:^(MASConstraintMaker *make) {
        make.width.mas_equalTo(82);
        make.height.mas_equalTo(26);
        make.left.offset(10);
        make.centerY.equalTo(self.compareButton);
    }];
    self.makeupOrLut.hidden = YES;
    
    /// 添加黑色背景父视图
    [self addSubview:self.blackView];
    [self.blackView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.top.equalTo(self.compareButton.mas_bottom).offset(10);
        make.left.right.bottom.offset(0);
    }];
    /// 添加重置
    [self.blackView addSubview:self.rightResetView];
    [self.rightResetView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.right.offset(-17);
        make.top.offset(2);
        make.width.mas_equalTo(47);
        make.height.mas_equalTo(36);
    }];
    /// 添加垂直分割线
    UIView *vLineView = [[UIView alloc] init];
    vLineView.backgroundColor = [UIColor colorWithWhite:1 alpha:0.1];
    [self.blackView addSubview:vLineView];
    [vLineView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.width.mas_equalTo(1);
        make.right.equalTo(self.rightResetView.mas_left).offset(-5);
        make.height.mas_equalTo(24);
        make.centerY.equalTo(self.rightResetView);
    }];
    /// 添加一级标题tab
    [self.blackView addSubview:self.titleCollection];
    [self.titleCollection mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.offset(0);
        make.top.offset(2);
        make.right.equalTo(vLineView.mas_left).offset(-5);//距离重置按钮左侧
        make.height.mas_equalTo(kTitleCollectionHeight);
    }];
    /// 添加水平分割线
    UIView *hLineView = [[UIView alloc] init];
    hLineView.backgroundColor = [TEUIConfig shareInstance].panelDividerColor;
    [self.blackView addSubview:hLineView];
    [hLineView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.width.mas_equalTo(self.mas_width);
        make.left.offset(0);
        make.top.equalTo(self.titleCollection.mas_bottom).offset(5);
        make.height.mas_equalTo(1);
    }];
    /// 添加二级标题tab
    [self.blackView addSubview:self.categoryCollection];
    [self.categoryCollection mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.right.offset(0);
        make.top.equalTo(hLineView.mas_bottom).offset(0);
        make.height.mas_equalTo(kTitleCollectionHeight);
    }];
    self.categoryCollection.hidden = YES;
    /// 添加特效视图
    [self.blackView addSubview:self.beautyCollection];
    [self.beautyCollection mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.offset(20);
        make.right.offset(-20);
        make.top.equalTo(self.titleCollection.mas_bottom).offset(15);
        make.bottom.offset(-30);
    }];
}

- (TESlider *)teSlider {
    if (!_teSlider) {
        _teSlider = [[TESlider alloc]init];
        [_teSlider setTintColor:[TEUIConfig shareInstance].seekBarProgressColor];
        _teSlider.minimumValue = 0;
        _teSlider.maximumValue = 100;
        [_teSlider addTarget:self action:@selector(valueChange:) forControlEvents:UIControlEventValueChanged];
        [_teSlider setThumbImage:[[TEUIConfig shareInstance] imageNamed:@"SliderIcon"] forState:UIControlStateNormal];
    }
    return _teSlider;
}

- (void)btnLongPress:(UILongPressGestureRecognizer *)gesture {
    if (gesture.state == UIGestureRecognizerStateBegan) {
        if (_showOrigin) {
            return;
        }
        if ([self.delegate respondsToSelector:@selector(showBeautyChanged:)]) {
            [self.delegate showBeautyChanged:NO];
        }
    } else if (gesture.state == UIGestureRecognizerStateEnded) {
        if (_showOrigin) {
            return;
        }
        if ([self.delegate respondsToSelector:@selector(showBeautyChanged:)]) {
            [self.delegate showBeautyChanged:YES];
        }
    }
}

- (void)setSubMenu:(BOOL)hide {
    if (!self.beautyTitleLabel) {
        self.beautyTitleLabel = [[UILabel alloc] init];
        [self.beautyTitleLabel setTextColor:[UIColor whiteColor]];
        self.beautyTitleLabel.font = [UIFont systemFontOfSize:16];
        [self.blackView addSubview:self.beautyTitleLabel];
    }
    CGFloat width;
    if ([TEUtils isCurrentLanguageHans]) {
        self.beautyTitleLabel.text = _currentUIProperty.displayName;
        width = [TEUtils textWidthFromTitle:_currentUIProperty.displayName font:self.beautyTitleLabel.font];
    } else {
        self.beautyTitleLabel.text = _currentUIProperty.displayNameEn;
        width = [TEUtils textWidthFromTitle:_currentUIProperty.displayNameEn font:self.beautyTitleLabel.font];
    }
    
    self.beautyTitleLabel.frame = CGRectMake((TEScreenWidth - width)/2, 5, width, 24);
    if (!_backButton) {
        _backButton = [[UIButton alloc] initWithFrame:CGRectMake(10, 2, 30, 30)];
        [_backButton setImage:[[TEUIConfig shareInstance] imageNamed:@"backto.png"] forState:UIControlStateNormal];
        [_backButton setBackgroundColor:[UIColor clearColor]];
        
        [self.blackView addSubview:_backButton];
        [_backButton addTarget:self action:@selector(clickBack) forControlEvents:UIControlEventTouchUpInside];
    }
    self.titleCollection.hidden = hide;
    self.backButton.hidden = !hide;
    self.beautyTitleLabel.hidden = !hide;
}

#pragma mark - Action
- (void)clickBack {
    TEUIProperty *selectProperty = [self getSelectedPoroerty:self.currentUIPropertyList];
    TEUIProperty *parent = [self getParentProperty:selectProperty property:self.currentUIProperty];
    if (selectProperty == parent) {
        [self setSubMenu:NO];
    } else {
        [self setSubMenu:YES];
    }
    self.currentUIProperty = parent;
    self.teSlider.hidden = YES;
    [self.beautyCollection reloadData];
    
    // 滚动到第一个 uiState == CHECKED_AND_IN_USE 的 property
    for (TEUIProperty *property in self.currentUIProperty.propertyList) {
        if (property.uiState == TEUIState_CHECKED_AND_IN_USE) {
            NSInteger index = [self.currentUIProperty.propertyList indexOfObject:property];
            if (index != NSNotFound) {
                NSIndexPath *indexPath = [NSIndexPath indexPathForItem:index inSection:0];
                [self.beautyCollection scrollToItemAtIndexPath:indexPath atScrollPosition:UICollectionViewScrollPositionLeft animated:NO];
            }
            break;
        }
    }
}

- (void)valueChange:(UISlider *)sender {
    self.teSlider.value = sender.value;
    if ((_currentUIProperty.teCategory == TECategory_MAKEUP || _currentUIProperty.teCategory == TECategory_LIGHTMAKEUP) && _makeupType == 1) {
        _curProperty.sdkParam.extraInfo.makeupLutStrength = [NSString stringWithFormat:@"%f",sender.value];
    } else {
        _curProperty.sdkParam.effectValue = sender.value;
    }
    [self.beautyProcess updateBeautyEffect:_curProperty];
}

- (UIView *)makeupOrLut {
    if (!_makeupOrLut) {
        self.makeupType = 0;
        weakify(self);
        _makeupOrLut = [self singleChoiceView:[[TEUIConfig shareInstance] localizedString:@"panel_makeup"] rightText:[[TEUIConfig shareInstance] localizedString:@"panel_lut"] clickLeft:^{
            strongify(self);
            self.makeupType = 0;
            self.teSlider.value = self.curProperty.sdkParam.effectValue;
        } rightClick:^{
            strongify(self);
            self.makeupType = 1;
            self.teSlider.value = [self.curProperty.sdkParam.extraInfo.makeupLutStrength intValue];
        } leftTag:120 select:self.makeupType];
    }
    return _makeupOrLut;
}

- (UIView *)rightResetView {
    if (!_rightResetView) {
        _rightResetView = [self rightView];
        UITapGestureRecognizer *tapGesture = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(resetClick)];
        [_rightResetView addGestureRecognizer:tapGesture];
    }
    return _rightResetView;
}

- (UIView *)rightView {
    UIView *rightView = [[UIView alloc] init];
    UIImageView *imageView = [[UIImageView alloc] init];
    [imageView setImage:[[TEUIConfig shareInstance] imageNamed:@"reset"]];
    [rightView addSubview:imageView];
    [imageView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.width.height.mas_equalTo(24);
        make.left.mas_equalTo(rightView);
        make.centerY.mas_equalTo(rightView.mas_centerY);
    }];
    UILabel *label = [[UILabel alloc] init];
    label.text = [[TEUIConfig shareInstance] localizedString:@"revert_btn_txt"];
    label.textColor = [UIColor whiteColor];
    label.textAlignment = NSTextAlignmentCenter;
    label.font = [UIFont fontWithName:@"PingFangSC-Regular" size:14];
    [rightView addSubview:label];
    CGFloat width = [TEUtils textWidthFromTitle:label.text font:label.font];
    [label mas_makeConstraints:^(MASConstraintMaker *make) {
        make.width.mas_equalTo(width);
        make.height.mas_equalTo(18);
        make.left.mas_equalTo(imageView.mas_right);
        make.centerY.mas_equalTo(imageView.mas_centerY);
    }];
    return rightView;
}

- (UIView *)singleChoiceView:(NSString *)leftText
                   rightText:(NSString *)rightText
                   clickLeft:(void(^)(void))clickLeft
                  rightClick:(void(^)(void))rightClick
                     leftTag:(NSInteger)leftTag
                      select:(NSInteger)select {
    UIView *view = [[UIView alloc] init];
    view.frame = CGRectMake(0, 0, 82, 26);
    view.backgroundColor = [UIColor colorWithWhite:1 alpha:0.4];
    view.layer.cornerRadius = 13;
    UIButton *leftBtn = [UIButton buttonWithType:UIButtonTypeCustom];
    [leftBtn setTitle:leftText forState:UIControlStateNormal];
    leftBtn.layer.cornerRadius = 13;
    leftBtn.tag = leftTag;
    [leftBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:0.4] forState:UIControlStateNormal];
    leftBtn.titleLabel.font = [UIFont fontWithName:@"PingFangSC-Regular" size:10];
    [leftBtn addTarget:self action:@selector(leftButtonAction:) forControlEvents:UIControlEventTouchUpInside];
    
    UIButton *rightBtn = [UIButton buttonWithType:UIButtonTypeCustom];
    [rightBtn setTitle:rightText forState:UIControlStateNormal];
    rightBtn.layer.cornerRadius = 13;
    rightBtn.tag = leftTag + 1;
    [rightBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:0.4] forState:UIControlStateNormal];
    rightBtn.titleLabel.font = [UIFont fontWithName:@"PingFangSC-Regular" size:10];
    [rightBtn addTarget:self action:@selector(rightButtonAction:) forControlEvents:UIControlEventTouchUpInside];
    objc_setAssociatedObject(leftBtn, "clickLeft", clickLeft, OBJC_ASSOCIATION_COPY_NONATOMIC);
    objc_setAssociatedObject(rightBtn, "rightClick", rightClick, OBJC_ASSOCIATION_COPY_NONATOMIC);
    
    if (select == 0) {
        [leftBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:1] forState:UIControlStateNormal];
        leftBtn.backgroundColor = [UIColor whiteColor];
        [rightBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:0.4] forState:UIControlStateNormal];
        rightBtn.backgroundColor = [UIColor clearColor];
    } else {
        [rightBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:1] forState:UIControlStateNormal];
        rightBtn.backgroundColor = [UIColor whiteColor];
        [leftBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:0.4] forState:UIControlStateNormal];
        leftBtn.backgroundColor = [UIColor clearColor];
    }
    [view addSubview:leftBtn];
    [leftBtn mas_makeConstraints:^(MASConstraintMaker *make) {
        make.width.mas_equalTo(41);
        make.height.mas_equalTo(26);
        make.top.mas_equalTo(view);
        make.left.mas_equalTo(view);
    }];
    [view addSubview:rightBtn];
    [rightBtn mas_makeConstraints:^(MASConstraintMaker *make) {
        make.width.mas_equalTo(41);
        make.height.mas_equalTo(26);
        make.top.mas_equalTo(view);
        make.right.mas_equalTo(view);
    }];
    return view;
}

- (void)leftButtonAction:(UIButton *)sender {
    // 获取关联的回调
    void (^clickLeft)(void) = objc_getAssociatedObject(sender, "clickLeft");
    // 修改 leftBtn 的背景颜色和文本颜色
    [sender setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:1] forState:UIControlStateNormal];
    sender.backgroundColor = [UIColor whiteColor];
    // 获取 rightBtn
    UIButton *rightBtn = [sender.superview viewWithTag:sender.tag + 1];
    // 修改 rightBtn 的背景颜色和文本颜色
    [rightBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:0.4] forState:UIControlStateNormal];
    rightBtn.backgroundColor = [UIColor clearColor];
    // 调用回调
    if (clickLeft) {
        clickLeft();
    }
}

- (void)rightButtonAction:(UIButton *)sender {
    // 获取关联的回调
    void (^rightClick)(void) = objc_getAssociatedObject(sender, "rightClick");
    // 修改 rightBtn 的背景颜色和文本颜色
    [sender setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:1] forState:UIControlStateNormal];
    sender.backgroundColor = [UIColor whiteColor];
    // 获取 leftBtn
    UIButton *leftBtn = [sender.superview viewWithTag:sender.tag -1];
    // 修改 leftBtn 的背景颜色和文本颜色
    [leftBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:0.4] forState:UIControlStateNormal];
    leftBtn.backgroundColor = [UIColor clearColor];
    // 调用回调
    if (rightClick) {
        rightClick();
    }
}

- (void)switchCollectionViewDirection {
    UICollectionViewScrollDirection scrollDirection = UICollectionViewScrollDirectionHorizontal;
    if (self.currentUIProperty.verticalLayout) {
        scrollDirection = UICollectionViewScrollDirectionVertical;
    }
    if (self.beautyLayout.scrollDirection != scrollDirection) {
        self.beautyLayout.scrollDirection = scrollDirection;
        [self.beautyCollection.collectionViewLayout invalidateLayout];
    }
}

- (void)resetClick {
    // 初始化UIAlertController
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"" message:@"" preferredStyle:UIAlertControllerStyleAlert];
    if (@available(iOS 13.0, *)) {
        alertController.overrideUserInterfaceStyle = UIUserInterfaceStyleLight;
    }
    //修改title字体及颜色
    NSMutableAttributedString *titleStr = [[NSMutableAttributedString alloc] initWithString:[[TEUIConfig shareInstance] localizedString:@"revert_tip_title"]];
    [titleStr addAttribute:NSForegroundColorAttributeName value:[UIColor colorWithRed:0/255.0 green:0/255.0 blue:0/255.0 alpha:0.9/1.0] range:NSMakeRange(0, titleStr.length)];
    [titleStr addAttribute:NSFontAttributeName value:[UIFont systemFontOfSize:20] range:NSMakeRange(0, titleStr.length)];
    [alertController setValue:titleStr forKey:@"attributedTitle"];
    // 修改message字体及颜色
    NSMutableAttributedString *messageStr = [[NSMutableAttributedString alloc] initWithString:[[TEUIConfig shareInstance] localizedString:@"revert_tip_msg"]];
    [messageStr addAttribute:NSForegroundColorAttributeName value:[UIColor colorWithRed:0/255.0 green:0/255.0 blue:0/255.0 alpha:0.3/1.0] range:NSMakeRange(0, messageStr.length)];
    [messageStr addAttribute:NSFontAttributeName value:[UIFont systemFontOfSize:18] range:NSMakeRange(0, messageStr.length)];
    [alertController setValue:messageStr forKey:@"attributedMessage"];
    // 添加UIAlertAction
    UIAlertAction *sureAction = [UIAlertAction actionWithTitle:[[TEUIConfig shareInstance] localizedString:@"revert_tip_dialog_right_btn"] style:UIAlertActionStyleDestructive handler:^(UIAlertAction * _Nonnull action) {
        [self performFullReset];
    }];
    // KVC修改字体颜色
    [sureAction setValue:[UIColor colorWithRed:0 green:0x6e/255.0 blue:1 alpha:1] forKey:@"_titleTextColor"];
    [alertController addAction:sureAction];
    UIAlertAction *cancelAction = [UIAlertAction actionWithTitle:[[TEUIConfig shareInstance] localizedString:@"revert_tip_dialog_left_btn"] style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        NSLog(@"取消");
    }];
    [cancelAction setValue:[UIColor blackColor] forKey:@"_titleTextColor"];
    [alertController addAction:cancelAction];
    [[self topViewController] presentViewController:alertController animated:YES completion:nil];
}

- (void)performFullReset {
    [self.tePanelDataProvider clearData];
    [self initData];
    [self updatePanelUI:self.currentUIProperty.hasSubTitle];
    [self switchCollectionViewDirection];
    [self setSubMenu:NO];
    [self.beautyProcess resetBeauty];
    [self reloadData];
    [self scrollToLeft];
    self.makeupOrLut.hidden = YES;
    self.teSlider.hidden = YES;
    [self.beautyProcess clearBeauty:[self.teBeautyKit getInUseSDKParamList]];
    [self setDefaultBeauty];
}

- (void)setDefaultBeauty {
    [self.beautyProcess setDefaultBeauty];
}

- (int)handleMediaAtPath:(NSString *)path {
    return [self.beautyProcess handleMediaAtPath:path];
}

- (void)openImagePicker {
    if ([self.delegate  respondsToSelector:@selector(onCustomSegBtnClick)]) {
        [self.delegate onCustomSegBtnClick];//获取客户自己创建的图片选择器，必须继承UINavigationController
    } else {
        dispatch_async(dispatch_get_main_queue(), ^{
            UIImagePickerController *picker = [[UIImagePickerController alloc] init];
            //资源类型为图片库
            picker.sourceType = UIImagePickerControllerSourceTypePhotoLibrary;
            picker.mediaTypes =@[(NSString*)kUTTypeMovie, (NSString*)kUTTypeImage];
            picker.delegate = self;
            //设置选择后的图片可被编辑
            picker.allowsEditing = NO;
            picker.modalPresentationStyle = UIModalPresentationFullScreen;
            [[self topViewController] presentViewController:picker animated:YES completion:nil];
        });
    }
}

- (void)showLoading {
    if (_isShowLoading) {
        return;
    }
    _isShowLoading = YES;
    UIViewController *curViewController = [self topViewController];
    [curViewController.view addSubview:self.loadingView];
    [self.loadingView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.top.mas_equalTo((curViewController.view.bounds.size.height -30)/2);
        make.left.mas_equalTo((curViewController.view.bounds.size.width -30)/2);
        make.width.height.mas_equalTo(30);
    }];
    [curViewController.view addSubview:self.processLabel];
    [self.processLabel mas_makeConstraints:^(MASConstraintMaker *make) {
        make.top.mas_equalTo(self.loadingView.mas_bottom).offset(3);
        make.centerX.mas_equalTo(self.loadingView.mas_centerX);
        make.width.mas_equalTo(200);
        make.height.mas_equalTo(30);
    }];
    [self.loadingView startAnimating];
}

- (void)dismissLoading {
    [self.loadingView stopAnimating];
    [self.loadingView removeFromSuperview];
    [self.processLabel removeFromSuperview];
    _isShowLoading = NO;
}

- (void)isShowCompareBtn:(BOOL)isShow {
    self.compareButton.hidden = !isShow;
    _showCompareBtn = isShow;
}

- (void)setExportParamList:(NSString *)lastParamList {
    if (!lastParamList) {
        return;
    }
    NSArray *beautyList = nil;
    id obj = [TEUtils jsonStringToJSONObject:lastParamList];
    if ([obj isKindOfClass:[NSArray class]]) {
        beautyList = (NSArray *)obj;
    }
    [self setUIState:self.currentUIPropertyList uiState:TEUIState_INIT];
    for (NSDictionary *dic in beautyList) {
        TESDKParam *param = [[TESDKParam alloc] initWithDict:dic];
        [self recoverBeautyData:self.currentUIPropertyList targetData:param];
    }
    [self setCurrentLevelUIState:self.currentUIPropertyList uiState:TEUIState_INIT];
    self.currentRootUIProperty.uiState = TEUIState_CHECKED_AND_IN_USE;
    dispatch_async(dispatch_get_main_queue(), ^{
        [self.beautyCollection reloadData];
    });
}

- (BOOL)recoverBeautyData:(NSMutableArray<TEUIProperty *>*)uiPropertyList targetData:(TESDKParam *)targetData {
    //step1: 从父级查找是否有需要设置的Effect，找到它
    //step2: 把它和它的父级的uiState设置成 1或2，并且调sdk设置效果
    for (TEUIProperty *uiproperty in uiPropertyList) {
        if (uiproperty.propertyList.count) {
            BOOL find = [self recoverBeautyData:uiproperty.propertyList targetData:targetData];
            if (find) {
                if ((uiproperty.teCategory == TECategory_BEAUTY ||
                     uiproperty.teCategory == TECategory_BEAUTYMAKEUP)) {
                    uiproperty.uiState = TEUIState_IN_USE;
                } else {
                    uiproperty.uiState = TEUIState_CHECKED_AND_IN_USE;
                }
                return YES;
            }
        } else {
            //到最底层了
            if (uiproperty.teCategory == TECategory_TEMPLATE &&
                [targetData.effectName isEqualToString:TEUI_BEAUTY_TEMPLATE] &&
                uiproperty.Id.intValue == targetData.effectValue) {
                [self.beautyProcess updateBeautyEffect:uiproperty];
                uiproperty.uiState = TEUIState_CHECKED_AND_IN_USE;
                return YES;
            } else if (uiproperty.teCategory == TECategory_BEAUTY &&
                       [uiproperty.sdkParam.effectName isEqualToString:targetData.effectName]) {
                //美颜
                uiproperty.sdkParam.effectValue = targetData.effectValue;
                [self.beautyProcess updateBeautyEffect:uiproperty];
                uiproperty.uiState = targetData.effectValue ? TEUIState_IN_USE : TEUIState_INIT;
                return YES;
            } else if (uiproperty.teCategory == TECategory_BEAUTYMAKEUP &&
                       targetData.resourcePath &&
                       [uiproperty.sdkParam.resourcePath containsString:targetData.resourcePath]) {
                //单点美妆
                uiproperty.sdkParam.effectValue = targetData.effectValue;
                [self.beautyProcess updateBeautyEffect:uiproperty];
                uiproperty.uiState = targetData.effectValue ? TEUIState_CHECKED_AND_IN_USE : TEUIState_INIT;
                return YES;
            } else if (targetData.resourcePath && [uiproperty.resourceUri containsString:targetData.resourcePath.lastPathComponent]) {
                uiproperty.resourceUri = targetData.resourcePath;
                uiproperty.sdkParam.effectValue = targetData.effectValue;
                uiproperty.sdkParam.extraInfo = [[ExtraInfo alloc] initWithDict:targetData.extraInfoDic];
                [self.beautyProcess updateBeautyEffect:uiproperty];
                uiproperty.uiState = TEUIState_CHECKED_AND_IN_USE;
                return YES;
            }
        }
    }
    return NO;
}

- (void)setEnhancedMode:(BOOL)enhancedMode {
    _enhancedMode = enhancedMode;
    if (_enhancedMode) {
        [self.teBeautyKit enableEnhancedMode:YES];
    }
    self.beautyProcess.enhancedMode=enhancedMode;
}

- (void)setUIState:(NSArray<TEUIProperty *>*)propertyList uiState:(int)uiState {
    for (TEUIProperty *property in propertyList) {
        property.uiState = uiState;
        [self setUIState:property.propertyList uiState:uiState];
    }
}

- (void)setCurrentUIProperty:(TEUIProperty *)currentUIProperty {
    if (_currentUIProperty != currentUIProperty) {
        _currentUIProperty = currentUIProperty;
        self.beautyProcess.currentUIProperty = currentUIProperty;
    }
}

- (void)setTePanelDataProvider:(TEPanelDataProvider *)tePanelDataProvider {
    if (_tePanelDataProvider != tePanelDataProvider) {
        _tePanelDataProvider = tePanelDataProvider;
        self.beautyProcess.tePanelDataProvider = tePanelDataProvider;
    }
}

- (void)setCurrentUIPropertyList:(NSMutableArray<TEUIProperty *> *)currentUIPropertyList {
    if (self.currentUIPropertyList != currentUIPropertyList) {
        _currentUIPropertyList = currentUIPropertyList;
        self.beautyProcess.currentUIPropertyList = currentUIPropertyList;
    }
}

- (NSArray<NSString *> *)searchMutualExclusion:(TEUIProperty *)selectedProperty {
    NSString *effectName = selectedProperty.sdkParam.effectName;
    NSArray *results;
    for (NSArray *exclusionArray in self.tePanelDataProvider.exclusionGroup) {
        //确定当前效果在哪个分组，只跟同组的效果互斥
        if ([exclusionArray containsObject:effectName]) {
            results = exclusionArray;
            break;
        }
    }
    return results;
}

#pragma mark - Private
//获取当前选中的property
- (TEUIProperty *)getSelectedPoroerty:(NSArray<TEUIProperty *> *)propertyList {
    TEUIProperty *defaultProperty = nil;
    BOOL hasCheck = NO;
    for (TEUIProperty *property in propertyList) {
        //有可能json里不止一个TEUIState_CHECKED_AND_IN_USE，要处理一下
        if (property.uiState == TEUIState_CHECKED_AND_IN_USE) {
            if (hasCheck) {
                property.uiState = TEUIState_INIT;
            } else {
                defaultProperty = property;
                hasCheck = YES;
            }
        }
    }
    return defaultProperty;
}
//取消选中美颜模板
- (void)deSelectBeautyTemplate:(TEUIProperty *)property forceAll:(BOOL)forceAll {
    if (forceAll ||
        property.teCategory == TECategory_BEAUTY ||
        property.teCategory == TECategory_LUT ||
        property.teCategory == TECategory_BEAUTYMAKEUP) {
        for (TEUIProperty *property in self.currentUIPropertyList) {
            if (property.teCategory == TECategory_TEMPLATE) {
                property.uiState = TEUIState_INIT;
                [self setUIState:property.propertyList uiState:TEUIState_INIT];
                break;
            }
        }
    }
}
//选中效果时，设置Check状态
- (void)setPropertyCheckState:(TEUIProperty *)property {
    if (property.uiState == TEUIState_CHECKED_AND_IN_USE) {
        if (property.propertyList.count) {
            BOOL inUse = NO;
            for (TEUIProperty *child in property.propertyList) {
                [self setPropertyCheckState:child];
                if (child.uiState == TEUIState_IN_USE) {
                    inUse = YES;
                }
            }
            property.uiState = inUse ? TEUIState_IN_USE : TEUIState_INIT;
        } else {
            property.uiState = property.sdkParam.effectValue ? TEUIState_IN_USE : TEUIState_INIT;
        }
    }
}
//设置uiState，本身和所有的上级
- (BOOL)setPropertyUIState:(TEUIProperty *)property
            parentProperty:(TEUIProperty *)parentProperty
                   uiState:(TEUIState)uiState {
    if ([parentProperty.propertyList containsObject:property]) {
        property.uiState = uiState;
        parentProperty.uiState = uiState;
        return YES;
    } else {
        for (TEUIProperty *child in parentProperty.propertyList) {
            BOOL find = [self setPropertyUIState:property parentProperty:child uiState:uiState];
            if (find) {
                parentProperty.uiState = uiState;
                return YES;
            }
        }
    }
    return NO;
}
//设置uiState，本身
- (void)setCurrentLevelUIState:(NSArray<TEUIProperty *>*)propertyList uiState:(TEUIState)uiState {
    for (TEUIProperty *property in propertyList) {
        property.uiState = uiState;
    }
}
//设置uiState，所有
- (void)setPropertyUIState:(NSArray<TEUIProperty *>*)propertyList uiState:(TEUIState)uiState {
    for (TEUIProperty *property in propertyList) {
        property.uiState = uiState;
        if (property.propertyList.count) {
            [self setPropertyUIState:property.propertyList uiState:uiState];
        }
    }
}
//更新tab
- (void)updatePanelUI:(BOOL)hasSubTitle {
    self.categoryCollection.hidden = !hasSubTitle;
    CGFloat verticalHeight = self.currentUIProperty.verticalLayout ? 60 : 0;
    //更新view高度
    [self.beautyCollection mas_updateConstraints:^(MASConstraintMaker *make) {
        make.top.equalTo(self.titleCollection.mas_bottom).offset(15 + (hasSubTitle ? kTitleCollectionHeight : 0));
    }];
    [self mas_updateConstraints:^(MASConstraintMaker *make) {
        make.height.mas_equalTo(kPanelViewHeight + (hasSubTitle ? kTitleCollectionHeight : 0) + verticalHeight);
    }];
}
//获取指定property的上级property
- (TEUIProperty *)getParentProperty:(TEUIProperty *)parentProperty property:(TEUIProperty *)property {
    if ([parentProperty.propertyList containsObject:property]) {
        return parentProperty;
    } else {
        for (TEUIProperty *child in parentProperty.propertyList) {
            TEUIProperty *parent = [self getParentProperty:child property:property];
            if (parent) {
                return parent;
            }
        }
    }
    return nil;
}

- (void)reloadData {
    [self.titleCollection reloadData];
    if (!self.categoryCollection.hidden) {
        [self.categoryCollection reloadData];
    }
    [self.beautyCollection reloadData];
}

- (void)scrollToLeft {
    NSIndexPath *indexPath = [NSIndexPath indexPathForItem:0 inSection:0];
    [self.titleCollection scrollToItemAtIndexPath:indexPath atScrollPosition:UICollectionViewScrollPositionLeft animated:NO];
    if (!self.categoryCollection.hidden) {
        [self.categoryCollection scrollToItemAtIndexPath:indexPath atScrollPosition:UICollectionViewScrollPositionLeft animated:NO];
    }
    [self.beautyCollection scrollToItemAtIndexPath:indexPath atScrollPosition:UICollectionViewScrollPositionLeft animated:NO];
}

- (void)checkUIState:(TEUIProperty *)property {
    if (property.propertyList.count) {
        for (TEUIProperty *child in property.propertyList) {
            [self checkUIState:child];
        }
    } else {
        if (property.uiState == TEUIState_CHECKED_AND_IN_USE) {
            [self updateSliderUI:property];
        }
    }
}

- (void)updateSliderUI:(TEUIProperty *)selectProperty {
    if (selectProperty.sdkParam.numericalType) {
        self.teSlider.hidden = NO;
        self.teSlider.minimumValue = [TESliderValueCalculator minValueForSliderWithModel:selectProperty];
        self.teSlider.maximumValue = [TESliderValueCalculator maxValueForSliderWithModel:selectProperty];
        self.teSlider.value = selectProperty.sdkParam.effectValue;
        if ((selectProperty.teCategory == TECategory_MAKEUP ||
             selectProperty.teCategory == TECategory_LIGHTMAKEUP) &&
            selectProperty.sdkParam.extraInfo.makeupLutStrength != nil) {
            self.makeupOrLut.hidden = NO;
            if (self.makeupType == 1) {
                self.teSlider.value = [selectProperty.sdkParam.extraInfo.makeupLutStrength intValue];
            }
            [self.teSlider mas_remakeConstraints:^(MASConstraintMaker *make) {
                make.left.mas_equalTo(self).mas_offset(102);
                make.right.mas_equalTo(self).mas_offset(-45);
                make.centerY.mas_equalTo(self.compareButton.mas_centerY);
            }];
        } else {
            self.makeupOrLut.hidden = YES;
            [self.teSlider mas_remakeConstraints:^(MASConstraintMaker *make) {
                make.left.mas_equalTo(self).mas_offset(10);
                make.right.mas_equalTo(self).mas_offset(-45);
                make.centerY.mas_equalTo(self.compareButton.mas_centerY);
            }];
        }
    } else {
        self.teSlider.hidden = YES;
        self.makeupOrLut.hidden = YES;
    }
}

#pragma mark - Alert
- (void)greenscreenAlert {
    if ([self.delegate  respondsToSelector:@selector(onGreenscreenItemClick)]) {
        [self.delegate onGreenscreenItemClick];
    } else {
        dispatch_async(dispatch_get_main_queue(), ^{
            UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"" message:@"" preferredStyle:UIAlertControllerStyleAlert];
            if (@available(iOS 13.0, *)) {
                alertController.overrideUserInterfaceStyle = UIUserInterfaceStyleLight;
            }
            NSMutableAttributedString *titleStr = [[NSMutableAttributedString alloc] initWithString:[[TEUIConfig shareInstance] localizedString:@"greenscreen_import_picture"]];
            [titleStr addAttribute:NSForegroundColorAttributeName value:[UIColor colorWithRed:0/255.0 green:0/255.0 blue:0/255.0 alpha:0.9/1.0] range:NSMakeRange(0, titleStr.length)];
            [titleStr addAttribute:NSFontAttributeName value:[UIFont systemFontOfSize:20] range:NSMakeRange(0, titleStr.length)];
            [alertController setValue:titleStr forKey:@"attributedTitle"];
            NSMutableAttributedString *messageStr = [[NSMutableAttributedString alloc] initWithString:[[TEUIConfig shareInstance] localizedString:@"greenscreen_msg"]];
            [messageStr addAttribute:NSForegroundColorAttributeName value:[UIColor colorWithRed:0/255.0 green:0/255.0 blue:0/255.0 alpha:0.3/1.0] range:NSMakeRange(0, messageStr.length)];
            [messageStr addAttribute:NSFontAttributeName value:[UIFont systemFontOfSize:18] range:NSMakeRange(0, messageStr.length)];
            [alertController setValue:messageStr forKey:@"attributedMessage"];
            UIAlertAction *sureAction = [UIAlertAction actionWithTitle:[[TEUIConfig shareInstance] localizedString:@"greenscreen_import_picture"] style:UIAlertActionStyleDestructive handler:^(UIAlertAction * _Nonnull action) {
                [self openImagePicker];
            }];
            [sureAction setValue:[UIColor colorWithRed:0 green:0x6e/255.0 blue:1 alpha:1] forKey:@"_titleTextColor"];
            [alertController addAction:sureAction];
            UIAlertAction *cancelAction = [UIAlertAction actionWithTitle:[[TEUIConfig shareInstance] localizedString:@"revert_tip_dialog_left_btn"] style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
                NSLog(@"取消");
            }];
            [cancelAction setValue:[UIColor blackColor] forKey:@"_titleTextColor"];
            [alertController addAction:cancelAction];
            [[self topViewController] presentViewController:alertController animated:YES completion:nil];
        });
    }
}

#pragma mark - lazyload
- (UIView *)blackView {
    if (!_blackView) {
        _blackView = [[UIView alloc] init];
        _blackView.backgroundColor = [TEUIConfig shareInstance].panelBackgroundColor;
    }
    return _blackView;
}

- (UICollectionView *)titleCollection {
    if (!_titleCollection) {
        _titleLayout = [[UICollectionViewFlowLayout alloc] init];
        _titleLayout.minimumInteritemSpacing = 20;
        _titleLayout.scrollDirection = UICollectionViewScrollDirectionHorizontal;
        _titleCollection = [[UICollectionView alloc] initWithFrame:CGRectZero collectionViewLayout:_titleLayout];
        _titleCollection.backgroundColor = [UIColor clearColor];
        _titleCollection.showsHorizontalScrollIndicator = NO;
        _titleCollection.contentInset = UIEdgeInsetsMake(0, 20, 0, 20);
        _titleCollection.delegate = self;
        _titleCollection.dataSource = self;
        [_titleCollection registerClass:[TETabCollectionViewCell class] forCellWithReuseIdentifier:kTETabCollectionViewCell];
    }
    return _titleCollection;
}

- (UICollectionView *)categoryCollection {
    if (!_categoryCollection) {
        _categoryLayout = [[UICollectionViewFlowLayout alloc] init];
        _categoryLayout.minimumInteritemSpacing = kCategoryTitleCollectionSpace;
        _categoryLayout.scrollDirection = UICollectionViewScrollDirectionHorizontal;
        _categoryCollection = [[UICollectionView alloc] initWithFrame:CGRectZero collectionViewLayout:_categoryLayout];
        _categoryCollection.backgroundColor = [UIColor clearColor];
        _categoryCollection.showsHorizontalScrollIndicator = NO;
        _categoryCollection.contentInset = UIEdgeInsetsMake(0, 20, 0, 20);
        _categoryCollection.delegate = self;
        _categoryCollection.dataSource = self;
        [_categoryCollection registerClass:[TETabCollectionViewCell class] forCellWithReuseIdentifier:kTETabCollectionViewCell];
    }
    return _categoryCollection;
}

- (UIButton *)compareButton {
    if (!_compareButton) {
        _compareButton = [UIButton buttonWithType:UIButtonTypeCustom];
        [_compareButton setImage:[[TEUIConfig shareInstance] imageNamed:@"compare.png"] forState:UIControlStateNormal];
        UILongPressGestureRecognizer *longPressRecognizer = [[UILongPressGestureRecognizer alloc] initWithTarget:self action:@selector(btnLongPress:)];
        longPressRecognizer.minimumPressDuration = 0; // 设置最短长按时间为 0，使其立即触发
        [_compareButton addGestureRecognizer:longPressRecognizer];
    }
    return _compareButton;
}

- (UICollectionView *)beautyCollection {
    if (!_beautyCollection) {
        _beautyLayout = [[UICollectionViewFlowLayout alloc] init];
        _beautyLayout.scrollDirection = UICollectionViewScrollDirectionHorizontal;
        _beautyCollection = [[UICollectionView alloc] initWithFrame:self.bounds collectionViewLayout:_beautyLayout];
        _beautyCollection.backgroundColor = [UIColor clearColor];
        _beautyCollection.dataSource = self;
        _beautyCollection.delegate = self;
        _beautyCollection.scrollEnabled = YES;
        _beautyCollection.showsHorizontalScrollIndicator = NO;
        _beautyCollection.showsVerticalScrollIndicator = NO;
        [_beautyCollection registerClass:[TECollectionViewCell class] forCellWithReuseIdentifier:kTECollectionViewCell];
    }
    return _beautyCollection;
}

- (TEBeautyProcess *)beautyProcess {
    if (!_beautyProcess) {
        _beautyProcess = [[TEBeautyProcess alloc] init];
        _beautyProcess.delegate = self;
        _beautyProcess.enhancedMode = self.enhancedMode;
    }
    _beautyProcess.teBeautyKit = self.teBeautyKit;
    return _beautyProcess;
}

- (UIActivityIndicatorView *)loadingView {
    if (!_loadingView) {
        _loadingView = [UIActivityIndicatorView new];
        _loadingView.color = [UIColor greenColor];
    }
    return  _loadingView;
}
- (UILabel *)processLabel {
    if (!_processLabel) {
        _processLabel = [UILabel new];
        _processLabel.textAlignment = NSTextAlignmentCenter;
        _processLabel.textColor = [UIColor whiteColor];
    }
    return  _processLabel;
}

#pragma mark - TEBeautyProcessDelegate
- (void)setEffect {
    if ([self.delegate respondsToSelector:@selector(setEffect)]) {
        [self.delegate setEffect];
    }
}

- (void)beautyCollectionRreloadData {
    [self.beautyCollection reloadData];
}

- (void)teSliderIsHidden {
    self.teSlider.hidden = YES;
}

- (void)teShowLoading {
    [self showLoading];
}

- (void)TEDownloaderProgressBlock:(CGFloat)progress {
    dispatch_async(dispatch_get_main_queue(), ^{
        self.showProgress = progress * 100;
        self.processLabel.text = [NSString stringWithFormat:@"%@%ld%%",[[TEUIConfig shareInstance] localizedString:@"downloading"],self.showProgress];
        if (self.showProgress == 100) {
            self.showProgress = 0;
            self.processLabel.text = @"";
        }
    });
}

- (void)teDismissLoading {
    dispatch_async(dispatch_get_main_queue(), ^{
        [self dismissLoading];
    });
}

- (void)teOpenImagePicker {
    [self openImagePicker];
}

- (void)teGreenscreenAlert {
    [self greenscreenAlert];
}

#pragma mark - UICollectionViewDelegate & UICollectionViewDataSource
- (NSInteger)collectionView:(UICollectionView *)collectionView numberOfItemsInSection:(NSInteger)section {
    //当前选中的属性
    TEUIProperty *property = self.currentUIProperty;
    if (collectionView == self.titleCollection) {
        return self.currentUIPropertyList.count;
    } else if (collectionView == self.categoryCollection) {
        return property.hasSubTitle ? property.propertyList.count : 0;
    } else {
        if (property.hasSubTitle) {
            return self.currentCategoryUIProperty.propertyList.count;
        } else {
            return property.propertyList.count;
        }
    }
}

- (UICollectionViewCell *)collectionView:(UICollectionView *)collectionView cellForItemAtIndexPath:(NSIndexPath *)indexPath {
    if (collectionView == self.titleCollection) {
        TEUIProperty *property = self.currentUIPropertyList[indexPath.row];
        TETabCollectionViewCell *cell = [collectionView dequeueReusableCellWithReuseIdentifier:kTETabCollectionViewCell forIndexPath:indexPath];
        NSString *title = [TEUtils isCurrentLanguageHans] ? property.displayName : property.displayNameEn;
        cell.titleLabel.text = title;
        if (property.uiState == TEUIState_CHECKED_AND_IN_USE) {
            cell.titleLabel.textColor = [UIColor whiteColor];
        } else {
            cell.titleLabel.textColor = [UIColor colorWithWhite:1 alpha:0.7];
        }
        return cell;
    } else if (collectionView == self.categoryCollection) {
        TEUIProperty *property = self.currentUIProperty.propertyList[indexPath.row];
        TETabCollectionViewCell *cell = [collectionView dequeueReusableCellWithReuseIdentifier:kTETabCollectionViewCell forIndexPath:indexPath];
        NSString *title = [TEUtils isCurrentLanguageHans] ? property.displayName : property.displayNameEn;
        cell.titleLabel.text = title;
        if (property.uiState == TEUIState_CHECKED_AND_IN_USE) {
            cell.titleLabel.textColor = [UIColor whiteColor];
        } else {
            cell.titleLabel.textColor = [UIColor colorWithWhite:1 alpha:0.7];
        }
        cell.titleLabel.font = [UIFont systemFontOfSize:14];
        return cell;
    } else {
        TECollectionViewCell *cell = [collectionView dequeueReusableCellWithReuseIdentifier:kTECollectionViewCell forIndexPath:indexPath];
        if (self.currentUIProperty.hasSubTitle) {
            cell.teUIProperty = self.currentCategoryUIProperty.propertyList[indexPath.row];
        } else {
            cell.teUIProperty = self.currentUIProperty.propertyList[indexPath.row];
        }
        cell.userInteractionEnabled = true;
        return cell;
    }
}

- (void)collectionView:(UICollectionView *)collectionView didSelectItemAtIndexPath:(NSIndexPath *)indexPath {
    [collectionView scrollToItemAtIndexPath:indexPath atScrollPosition:UICollectionViewScrollPositionCenteredHorizontally animated:YES];
    if (collectionView == self.titleCollection) {
        TEUIProperty *property = self.currentUIPropertyList[indexPath.row];
        [self setCurrentLevelUIState:self.currentUIPropertyList uiState:TEUIState_INIT];
        property.uiState = TEUIState_CHECKED_AND_IN_USE;
        //记录当前选中的状态
        self.currentUIProperty = property;
        self.currentRootUIProperty = property;
        self.teSlider.hidden = YES;
        self.makeupOrLut.hidden = YES;
        if (property.hasSubTitle) {
            //包含二级标题，找到二级标题选中的属性
            self.currentCategoryUIProperty = [self getSelectedPoroerty:property.propertyList];
            if (!self.currentCategoryUIProperty) {
                self.currentCategoryUIProperty = property.propertyList.firstObject;
                self.currentCategoryUIProperty.uiState = TEUIState_CHECKED_AND_IN_USE;
            }
        }
        [self switchCollectionViewDirection];
        [self updatePanelUI:property.hasSubTitle];
        [self checkUIState:property];
        [self.titleCollection reloadData];
        [self.categoryCollection reloadData];
        [self.beautyCollection reloadData];
    } else if (collectionView == self.categoryCollection) {
        TEUIProperty *property = self.currentUIProperty.propertyList[indexPath.row];
        [self setCurrentLevelUIState:self.currentUIProperty.propertyList uiState:TEUIState_INIT];
        property.uiState = TEUIState_CHECKED_AND_IN_USE;
        self.currentCategoryUIProperty = property;
        self.teSlider.hidden = YES;
        self.makeupOrLut.hidden = YES;
        [self checkUIState:property];
        [self.categoryCollection reloadData];
        [self.beautyCollection reloadData];
    } else {
        if (self.makeupType == 1) {
            self.makeupType = 0;
            UIButton *leftBtn = [self.makeupOrLut viewWithTag:120];
            [leftBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:1] forState:UIControlStateNormal];
            leftBtn.backgroundColor = [UIColor whiteColor];
            UIButton *rightBtn = [self.makeupOrLut viewWithTag:120 + 1];
            [rightBtn setTitleColor:[UIColor colorWithRed:0 green:0 blue:0 alpha:0.4] forState:UIControlStateNormal];
            rightBtn.backgroundColor = [UIColor clearColor];
        }
        self.greenBackgroundV2Property = nil;
        TEUIProperty *selectTabProperty = self.currentUIProperty;
        if (self.currentUIProperty.hasSubTitle) {
            selectTabProperty = self.currentCategoryUIProperty;
        }
        TEUIProperty *selectProperty = selectTabProperty.propertyList[indexPath.row];
        if (selectProperty.propertyList.count == 0) {
            for (TEUIProperty *property in selectTabProperty.propertyList) {
                //检查互斥
                if ([[self searchMutualExclusion:property] containsObject:selectProperty.sdkParam.effectName]) {
                    property.uiState = TEUIState_INIT;
                }
            }
            [self deSelectBeautyTemplate:selectProperty forceAll:NO];
            [self setPropertyCheckState:self.currentRootUIProperty];
            [self setPropertyUIState:selectProperty parentProperty:self.currentRootUIProperty uiState:TEUIState_CHECKED_AND_IN_USE];
            [self updateSliderUI:selectProperty];
            self.curProperty = selectProperty;
            [self.beautyProcess updateBeautyEffect:_curProperty];
            if ([self.delegate respondsToSelector:@selector(selectEffect:)]) {
                [self.delegate selectEffect:_curProperty];
            }
            [self.beautyCollection reloadData];
        } else {
            if (selectProperty.teCategory == TECategory_SEGMENTATION) {
                self.greenBackgroundV2Property = selectProperty;
                [self.beautyProcess updateBeautyEffect:selectProperty];
            } else {
                [self showSubEffectPanelView:selectProperty];
            }
        }
    }
}

- (CGSize)collectionView:(UICollectionView *)collectionView layout:(UICollectionViewLayout *)collectionViewLayout sizeForItemAtIndexPath:(NSIndexPath *)indexPath {
    if (collectionView == self.titleCollection) {
        NSString *title = [TEUtils isCurrentLanguageHans] ? self.currentUIPropertyList[indexPath.row].displayName : self.currentUIPropertyList[indexPath.row].displayNameEn;
        CGFloat textWidth = [TEUtils textWidthFromTitle:title font:[UIFont systemFontOfSize:16]];
        return CGSizeMake(textWidth + 5, 36);
    } else if (collectionView == self.categoryCollection) {
        return CGSizeMake(kCategoryTitleCollectionItemWidth, 36);
    }
    return CGSizeMake(beautyCollectionItemWidth, beautyCollectionItemHeight);
}

#pragma mark - 打开相册
- (void)imagePickerController:(UIImagePickerController *)picker didFinishPickingMediaWithInfo:(NSDictionary<NSString *,id> *)info {
    // 移除相册界面
    [picker.view removeFromSuperview];
    // 获取文件类型:
    NSString *mediaType = info[UIImagePickerControllerMediaType];
    if ([mediaType isEqualToString:(NSString*)kUTTypeImage]) {
        // 用户选的文件为图片类型(kUTTypeImage)
        UIImage *image = info[UIImagePickerControllerOriginalImage];
        [self updateSegmentationUI:YES error:nil];
        [self.beautyProcess imagePickerFinish:image picker:picker];
    } else if([mediaType isEqualToString:(NSString*)kUTTypeMovie]){
        NSURL *sourceURL = [info objectForKey:UIImagePickerControllerMediaURL];
        if(sourceURL){
            weakify(self); // 使用 weakify 避免循环引用
            [self.beautyProcess moviePickerFinish:sourceURL picker:picker completionHandler:^(BOOL success, NSError * _Nullable error, NSInteger timeOffset) {
                strongify(self);
                [self updateSegmentationUI:success error:error];
            }];
        }
    }
}

- (void)updateSegmentationUI:(BOOL)success error:(NSError *)error {
    if (success) {
        [self showSubEffectPanelView:self.greenBackgroundV2Property];
    } else {
        dispatch_async(dispatch_get_main_queue(), ^{
            [self showErrorAlert:(int)error.code];
        });
    }
}

- (void)showErrorAlert:(int)errorCode {
    if (errorCode == 0) {
        return;
    }
    NSString* errorMsg = @"";
    switch (errorCode) {
        case 5000:
            errorMsg = @"分割背景图片分辨率超过2160*3840";
            break;
        case 5002:
            errorMsg = @"分割背景视频解析失败";
            break;
        case 5003:
            errorMsg = @"分割背景视频超过200秒";
            break;
        case 5004:
            errorMsg = @"分割背景视频格式不支持";
            break;
        default:
            break;
    }
    
    UIAlertController *alertVC = [UIAlertController alertControllerWithTitle:@"背景导入失败" message:[NSString stringWithFormat:@"%i: %@", errorCode, errorMsg] preferredStyle:UIAlertControllerStyleAlert];
    [[self topViewController] presentViewController:alertVC animated:YES completion:nil];
    [self performSelector:@selector(dismissAlert:) withObject:alertVC afterDelay:2.0];
}

- (void)dismissAlert:(UIAlertController *)alert {
    [alert dismissViewControllerAnimated:YES completion:nil];
}

- (void)showSubEffectPanelView:(TEUIProperty *)property {
    if (property) {
        self.teSlider.hidden = YES;
        self.makeupOrLut.hidden = YES;
        self.currentUIProperty = property;
        [self setSubMenu:YES];
        [self.beautyCollection reloadData];
        for (TEUIProperty *subProperty in property.propertyList) {
            if (subProperty.propertyList.count == 0 && subProperty.uiState == TEUIState_CHECKED_AND_IN_USE) {
                [self updateSliderUI:subProperty];
                break;
            }
        }
    }
}

// 取消图片选择回调
- (void)imagePickerControllerDidCancel:(UIImagePickerController *)picker {
    [picker dismissViewControllerAnimated:YES completion:nil];
    NSLog(@"取消");
}

- (UIViewController *)topViewController {
    UIWindow *keyWindow = [self getKeyWindow];
    UIViewController *rootVC = keyWindow.rootViewController;
    return [self topViewControllerFrom:rootVC];
}

// 递归遍历找到最上层控制器
- (UIViewController *)topViewControllerFrom:(UIViewController *)vc {
    if ([vc isKindOfClass:[UINavigationController class]]) {
        UINavigationController *nav = (UINavigationController *)vc;
        return [self topViewControllerFrom:nav.visibleViewController];
    } else if ([vc isKindOfClass:[UITabBarController class]]) {
        UITabBarController *tab = (UITabBarController *)vc;
        return [self topViewControllerFrom:tab.selectedViewController];
    } else if (vc.presentedViewController) {
        return [self topViewControllerFrom:vc.presentedViewController];
    } else {
        return vc;
    }
}

// iOS13+ 获取 keyWindow 的推荐写法
- (UIWindow *)getKeyWindow {
    if (@available(iOS 13.0, *)) {
        // 使用 scenes 获取当前 activity window
        NSSet *scenes = [UIApplication sharedApplication].connectedScenes;
        for (UIScene *scene in scenes) {
            if (scene.activationState == UISceneActivationStateForegroundActive &&
                [scene isKindOfClass:[UIWindowScene class]]) {
                UIWindowScene *windowScene = (UIWindowScene *)scene;
                for (UIWindow *window in windowScene.windows) {
                    if (window.isKeyWindow) {
                        return window;
                    }
                }
            }
        }
        // 如果没找到激活的 window，返回第一个 window 备选
        return [UIApplication sharedApplication].windows.firstObject;
    } else {
        // iOS 13 以下
        return [UIApplication sharedApplication].keyWindow;
    }
}

- (void)dealloc {
    [_tePanelDataProvider clearData];
}

@end
