package com.tencent.ugcupload.videoupload;


import android.content.Context;
import android.graphics.Bitmap;
import android.media.MediaMetadataRetriever;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.text.TextUtils;
import android.util.Log;
import android.webkit.MimeTypeMap;


import com.tencent.ugcupload.videoupload.impl.TVCClient;
import com.tencent.ugcupload.videoupload.impl.TVCConfig;
import com.tencent.ugcupload.videoupload.impl.TVCConstants;
import com.tencent.ugcupload.videoupload.impl.TVCLog;
import com.tencent.ugcupload.videoupload.impl.TVCUploadInfo;
import com.tencent.ugcupload.videoupload.impl.TVCUploadListener;
import com.tencent.ugcupload.videoupload.impl.TVCUtils;
import com.tencent.ugcupload.videoupload.impl.TXUGCPublishOptCenter;
import com.tencent.ugcupload.videoupload.impl.UGCReport;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URLConnection;


/**
 * Short Video Publishing Interface Class
 * 短视频发布接口类
 */
public class TXUGCPublish {
    public static boolean isDebug = true;
    private static final String TAG = "TXVideoPublish";
    private static final long COVER_TIME = 500 * 1000;
    private Context mContext;
    private Handler mHandler;
    private TXUGCPublishTypeDef.ITXVideoPublishListener mListener;
    private TXUGCPublishTypeDef.ITXMediaPublishListener mMediaListener;
    private boolean mPublishing;
    private TVCClient mTVCClient = null;
    private String mCustomKey = "";
    private boolean mIsCancel = false;
    private final String mUploadKey;

    public TXUGCPublish(Context context, String customKey) {
        this(context, customKey, "");
    }

    public TXUGCPublish(Context context, String customKey, String uploadKey) {
        mCustomKey = customKey;
        mUploadKey = uploadKey;
        Log.i(TAG, "create new TXUGCPublish, context:" + context + ",customKey:"
                + customKey + ",uploadKey:" + uploadKey);
        if (context != null) {
            mContext = context;
            mHandler = new Handler(mContext.getMainLooper());
            setIsDebug(true);
        }
    }

    public TXUGCPublish(Context context) {
        this(context, "");
    }

    public void setListener(TXUGCPublishTypeDef.ITXVideoPublishListener listener) {
        mListener = listener;
    }

    public void setListener(TXUGCPublishTypeDef.ITXMediaPublishListener listener) {
        mMediaListener = listener;
    }

    /**
     * Set whether to print logs
     * 设置是否打印日志
     */
    public void setIsDebug(boolean isDebug) {
        TXUGCPublish.isDebug = isDebug;
        TVCLog.setDebuggable(isDebug, mContext);
        TVCLog.i(TAG, "setIsDebug:" + isDebug);
    }

    private int publishVideoImpl(TXUGCPublishTypeDef.TXPublishParam param) {
        if (TextUtils.isEmpty(param.videoPath)) {
            TVCLog.e(TAG, "publishVideo invalid videoPath");
            return TVCConstants.ERR_UGC_INVALID_VIDOPATH;
        }

        boolean bVideoFileExist = TVCUtils.isExistsForPathOrUri(mContext, param.videoPath);

        if (!bVideoFileExist) {
            TVCLog.e(TAG, "publishVideo invalid video file");
            return TVCConstants.ERR_UGC_INVALID_VIDEO_FILE;
        }

        String coverPath = "";
        if (!TextUtils.isEmpty(param.coverPath)) {
            coverPath = param.coverPath;
            File file = new File(coverPath);
            if (!file.exists()) {
                return TVCConstants.ERR_UGC_INVALID_COVER_PATH;
            }
        }

        TVCConfig tvcConfig = new TVCConfig();
        tvcConfig.mCustomKey = mCustomKey;
        tvcConfig.mSignature = param.signature;
        tvcConfig.mEnableResume = param.enableResume;
        tvcConfig.mEnableHttps = param.enableHttps;
        tvcConfig.mVodReqTimeOutInSec = 10;
        tvcConfig.mSliceSize = param.sliceSize;
        tvcConfig.mConcurrentCount = param.concurrentCount;
        tvcConfig.mTrafficLimit = param.trafficLimit;
        tvcConfig.mUploadResumeController = param.uploadResumeController;
        tvcConfig.mIsDebuggable = isDebug;

        if (mTVCClient == null) {
            mTVCClient = new TVCClient(mContext, tvcConfig, mUploadKey);
        } else {
            mTVCClient.updateConfig(tvcConfig);
        }

        final TVCUploadInfo info = new TVCUploadInfo(getFileType(param.videoPath), param.videoPath,
                getFileType(coverPath),
                coverPath, param.fileName);

        if (info.getFileSize() == 0) {
            TVCLog.e(TAG, "publishVideo invalid videoPath");
            return TVCConstants.ERR_UGC_INVALID_VIDOPATH;
        }

        final long upLoadStartTime = System.currentTimeMillis();
        int ret = mTVCClient.uploadVideo(info, new TVCUploadListener() {
            @Override
            public void onSuccess(final String fileId, final String playUrl, final String coverUrl) {
                if (mHandler != null) {
                    mHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (mListener != null) {
                                TXUGCPublishTypeDef.TXPublishResult result = new TXUGCPublishTypeDef.TXPublishResult();
                                result.retCode = TXUGCPublishTypeDef.PUBLISH_RESULT_OK;
                                result.descMsg = "publish success";
                                result.videoId = fileId;
                                result.videoURL = playUrl;
                                result.coverURL = coverUrl;
                                mListener.onPublishComplete(result);
                            }
                            TVCLog.i(TAG, "upload cost Time:" + (System.currentTimeMillis() - upLoadStartTime));
                        }
                    });
                }
                mTVCClient = null;
                mPublishing = false;
            }

            @Override
            public void onFailed(final int errCode, final String errMsg) {
                if (mHandler != null) {
                    mHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (mListener != null) {
                                TXUGCPublishTypeDef.TXPublishResult result = new TXUGCPublishTypeDef.TXPublishResult();
                                result.retCode = errCode;
                                result.descMsg = errMsg;
                                mListener.onPublishComplete(result);
                                TVCLog.i(TAG, "upload onFailed，errorMsg:"
                                        + errMsg + ",t:" + (System.currentTimeMillis() - upLoadStartTime));
                            }
                        }
                    });
                }
                mTVCClient = null;
                mPublishing = false;
            }

            @Override
            public void onProgress(final long currentSize, final long totalSize) {
                if (mHandler != null) {
                    mHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (mListener != null) {
                                mListener.onPublishProgress(currentSize, totalSize);
                            }
                        }
                    });
                }
                mPublishing = false;
            }
        });
        return ret;
    }

    /**
     * Upload video file (video file + cover image)
     * 上传视频文件 （视频文件 + 封面图）
     */
    public int publishVideo(final TXUGCPublishTypeDef.TXPublishParam param) {
        TVCLog.i(TAG, "vodPublish version:" + TVCConstants.TVCVERSION);
        if (mPublishing) {
            TVCLog.e(TAG, "there is existing publish task");
            return TVCConstants.ERR_UGC_PUBLISHING;
        }

        if (param == null) {
            TVCLog.e(TAG, "publishVideo invalid param");
            return TVCConstants.ERR_UGC_INVALID_PARAM;
        }
        if (TextUtils.isEmpty(param.signature)) {
            TVCLog.e(TAG, "publishVideo invalid UGCSignature");
            return TVCConstants.ERR_UGC_INVALID_SIGNATURE;
        }
        mPublishing = true;
        mIsCancel = false;
        if (param.enablePreparePublish) {
            // Start pre-publishing initialization, start uploading after pre-publishing.
            TXUGCPublishOptCenter.getInstance().prepareUpload(mContext, param.signature,
                    new TXUGCPublishOptCenter.IPrepareUploadCallback() {
                        @Override
                        public void onFinish() {
                            if (mIsCancel) {
                                mIsCancel = false;
                                TVCLog.i(TAG, "upload is cancel after prepare upload");
                                TXUGCPublishTypeDef.TXPublishResult result = new TXUGCPublishTypeDef.TXPublishResult();
                                result.retCode = TVCConstants.ERR_USER_CANCEL;
                                result.descMsg = "request is cancelled by manual pause";
                                mListener.onPublishComplete(result);
                                return;
                            }
                            int ret = publishVideoImpl(param);
                            if (ret != TVCConstants.NO_ERROR) {
                                TVCLog.e(TAG, "upload params check failed:" + ret);
                                callbackErrorComplete(ret, "upload params check failed");
                            } else {
                                mPublishing = true;
                            }
                        }
                    });
            return TVCConstants.NO_ERROR;
        } else {
            TXUGCPublishOptCenter.getInstance().prepareUpload(mContext, param.signature, null);
            int ret = publishVideoImpl(param);
            if (ret != TVCConstants.NO_ERROR) {
                TVCLog.e(TAG, "upload params check failed:" + ret);
                callbackErrorComplete(ret, "upload params check failed");
            } else {
                mPublishing = true;
            }
            return ret;
        }
    }

    private int publishMediaImpl(TXUGCPublishTypeDef.TXMediaPublishParam param) {
        if (TextUtils.isEmpty(param.mediaPath)) {
            TVCLog.e(TAG, "publishVideo invalid videoPath");
            return TVCConstants.ERR_UGC_INVALID_VIDOPATH;
        }

        boolean bVideoFileExist = false;
        try {
            File file = new File(param.mediaPath);
            bVideoFileExist = file.isFile() && file.exists();
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (!bVideoFileExist) {
            TVCLog.e(TAG, "publishVideo invalid video file");
            return TVCConstants.ERR_UGC_INVALID_VIDEO_FILE;
        }

        TVCConfig tvcConfig = new TVCConfig();
        tvcConfig.mCustomKey = mCustomKey;
        tvcConfig.mSignature = param.signature;
        tvcConfig.mEnableResume = param.enableResume;
        tvcConfig.mEnableHttps = param.enableHttps;
        tvcConfig.mVodReqTimeOutInSec = 10;
        tvcConfig.mSliceSize = param.sliceSize;
        tvcConfig.mConcurrentCount = param.concurrentCount;
        tvcConfig.mTrafficLimit = param.trafficLimit;
        tvcConfig.mUploadResumeController = param.uploadResumeController;
        tvcConfig.mIsDebuggable = isDebug;

        if (mTVCClient == null) {
            mTVCClient = new TVCClient(mContext, tvcConfig, mUploadKey);
        } else {
            mTVCClient.updateConfig(tvcConfig);
        }

        TVCUploadInfo info = new TVCUploadInfo(getFileType(param.mediaPath), param.mediaPath, null, null,
                param.fileName);

        if (info.getFileSize() == 0) {
            TVCLog.e(TAG, "publishVideo invalid videoPath");
            return TVCConstants.ERR_UGC_INVALID_VIDOPATH;
        }

        final long upLoadStartTime = System.currentTimeMillis();
        int ret = mTVCClient.uploadVideo(info, new TVCUploadListener() {
            @Override
            public void onSuccess(final String fileId, final String playUrl, final String coverUrl) {
                if (mHandler != null) {
                    mHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (mMediaListener != null) {
                                TXUGCPublishTypeDef.TXMediaPublishResult result =
                                        new TXUGCPublishTypeDef.TXMediaPublishResult();
                                result.retCode = TXUGCPublishTypeDef.PUBLISH_RESULT_OK;
                                result.descMsg = "publish success";
                                result.mediaId = fileId;
                                result.mediaURL = playUrl;
                                mMediaListener.onMediaPublishComplete(result);
                            }
                            TVCLog.i(TAG, "upload cost Time:" + (System.currentTimeMillis() - upLoadStartTime));
                        }
                    });
                }
                mTVCClient = null;
                mPublishing = false;
            }

            @Override
            public void onFailed(final int errCode, final String errMsg) {
                if (mHandler != null) {
                    mHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (mMediaListener != null) {
                                TXUGCPublishTypeDef.TXMediaPublishResult result =
                                        new TXUGCPublishTypeDef.TXMediaPublishResult();
                                result.retCode = errCode;
                                result.descMsg = errMsg;
                                mMediaListener.onMediaPublishComplete(result);
                            }
                        }
                    });
                }
                mTVCClient = null;
                mPublishing = false;
            }

            @Override
            public void onProgress(final long currentSize, final long totalSize) {
                if (mHandler != null) {
                    mHandler.post(new Runnable() {
                        @Override
                        public void run() {
                            if (mMediaListener != null) {
                                mMediaListener.onMediaPublishProgress(currentSize, totalSize);
                            }
                        }
                    });
                }
                mPublishing = false;
            }
        });
        return ret;
    }

    /**
     * Upload media file
     * 上传媒体文件
     */
    public int publishMedia(final TXUGCPublishTypeDef.TXMediaPublishParam param) {
        TVCLog.i(TAG, "vodPublish version:" + TVCConstants.TVCVERSION);
        if (mPublishing) {
            TVCLog.e(TAG, "there is existing publish task");
            return TVCConstants.ERR_UGC_PUBLISHING;
        }
        if (param == null) {
            TVCLog.e(TAG, "publishVideo invalid param");
            return TVCConstants.ERR_UGC_INVALID_PARAM;
        }
        if (TextUtils.isEmpty(param.signature)) {
            TVCLog.e(TAG, "publishVideo invalid UGCSignature");
            return TVCConstants.ERR_UGC_INVALID_SIGNATURE;
        }
        mPublishing = true;
        mIsCancel = false;
        if (param.enablePreparePublish) {
            TXUGCPublishOptCenter.getInstance().prepareUpload(mContext, param.signature,
                    new TXUGCPublishOptCenter.IPrepareUploadCallback() {
                        @Override
                        public void onFinish() {
                            if (mIsCancel) {
                                mIsCancel = false;
                                TVCLog.i(TAG, "upload is cancel after prepare upload");
                                callbackErrorComplete(TVCConstants.ERR_USER_CANCEL, "request is cancelled by manual pause");
                                return;
                            }
                            int ret = publishMediaImpl(param);
                            if (ret != TVCConstants.NO_ERROR) {
                                TVCLog.e(TAG, "upload params check failed:" + ret);
                                callbackErrorComplete(ret, "upload params check failed");
                            } else {
                                mPublishing = true;
                            }
                        }
                    });
            return TVCConstants.NO_ERROR;
        } else {
            TXUGCPublishOptCenter.getInstance().prepareUpload(mContext, param.signature, null);
            int ret = publishMediaImpl(param);
            if (ret != TVCConstants.NO_ERROR) {
                TVCLog.e(TAG, "upload params check failed:" + ret);
                callbackErrorComplete(ret, "upload params check failed");
            } else {
                mPublishing = true;
            }
            return ret;
        }
    }

    /**
     * Set on-demand appId
     * Its function is to facilitate the location of problems that occur during the upload process.
     * 设置点播appId
     * 作用是方便定位上传过程中出现的问题
     */
    public void setAppId(int appId) {
        if (mTVCClient != null) {
            mTVCClient.setAppId(appId);
        }
    }

    /**
     * Cancel upload (cancel media/cancel short video publishing)
     * Note: What is cancelled are the unstarted fragments. If the uploaded source file is too small, there are no
     * fragments left to trigger the upload when cancelling, and the final file will still be uploaded completely.
     * 取消上传 （取消媒体/取消短视频发布）
     * 注意：取消的是未开始的分片。如果上传源文件太小，取消的时候已经没有分片还未触发上传，最终文件还是会上传完成
     */
    public void canclePublish() {
        if (mTVCClient != null) {
            mTVCClient.cancelUpload();
        }
        mPublishing = false;
    }

    /**
     * Get reporting information
     * 获取上报信息
     */
    public Bundle getStatusInfo() {
        if (mTVCClient != null) {
            return mTVCClient.getStatusInfo();
        } else {
            return null;
        }
    }


    private String getFileType(String filePath) {
        String fileType = "";
        if (TextUtils.isEmpty(filePath)) {
            return fileType;
        }
        if (filePath.startsWith("content://")) {
            fileType = getFileTypeByUri(Uri.parse(filePath));
        }
        if (TextUtils.isEmpty(fileType)) {
            String absolutePath = TVCUtils.getAbsolutePath(mContext, filePath);
            fileType = getFileTypeByURL(absolutePath);
        }
        return fileType;
    }

    private String getFileTypeByURL(String filePath) {
        String fileType = "";
        File file = new File(filePath);
        if (file.exists() && file.canRead()) {
            try {
                URLConnection urlConnection = file.toURI().toURL().openConnection();
                String mimeType = urlConnection.getContentType();
                if (!TextUtils.isEmpty(mimeType)) {
                    fileType = MimeTypeMap.getSingleton().getExtensionFromMimeType(mimeType);
                }
            } catch (IOException e) {
                TVCLog.e(TAG, "getFileTypeByURL failed, path:" + filePath + ", error:" + e);
            }
        }
        return fileType;
    }

    private String getFileTypeByUri(Uri uri) {
        String fileType = "";
        if (null != uri) {
            fileType = mContext.getContentResolver().getType(uri);
            if (!TextUtils.isEmpty(fileType)) {
                int index = fileType.lastIndexOf("/");
                if (index != -1) {
                    fileType = fileType.substring(index + 1);
                }
            }
        }
        return fileType;
    }


    private String getVideoThumb(String videoPath) {
        String strCoverFilePath = null;
        try {
            File videoFile = new File(videoPath);
            if (!videoFile.exists()) {
                TVCLog.w(TAG, "record: video file is not exists when record finish");
                return null;
            }
            MediaMetadataRetriever media = new MediaMetadataRetriever();
            media.setDataSource(videoPath);
            Bitmap thumb = media.getFrameAtTime(COVER_TIME);

            String fileName = "";
            int index = videoPath.lastIndexOf(".");
            if (index != -1) {
                fileName = videoPath.substring(0, index);
            }

            strCoverFilePath = fileName + ".jpg";
            File f = new File(strCoverFilePath);
            if (f.exists()) f.delete();
            FileOutputStream fOut = null;
            fOut = new FileOutputStream(f);
            thumb.compress(Bitmap.CompressFormat.JPEG, 100, fOut);
            fOut.flush();
            fOut.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return strCoverFilePath;
    }

    private void callbackErrorComplete(int code, String msg) {
        mPublishing = false;
        TXUGCPublishTypeDef.TXPublishResult result = new TXUGCPublishTypeDef.TXPublishResult();
        result.retCode = code;
        result.descMsg = msg;
        mListener.onPublishComplete(result);
    }

    /**
     * set host env type
     *
     * @param envType {@link TVCConstants#ENV_DOMESTIC},{@link TVCConstants#ENV_INTL}
     */
    public static void setEnvConfig(int envType) {
        if (envType == TVCConstants.ENV_DOMESTIC) {
            TVCConstants.ENV_TYPE = envType;
            TVCConstants.VOD_SERVER_HOST = TVCConstants.VOD_SERVER_DOMESTIC_HOST;
            TVCConstants.VOD_SERVER_HOST_BAK = TVCConstants.VOD_SERVER_DOMESTIC_HOST_BAK;
            UGCReport.REPORT_URL = TVCConstants.VOD_REPORT_DOMESTIC_HOST;
            UGCReport.REPORT_URL_BAK = TVCConstants.VOD_REPORT_DOMESTIC_HOST_BAK;
        } else if (envType == TVCConstants.ENV_INTL) {
            TVCConstants.ENV_TYPE = envType;
            TVCConstants.VOD_SERVER_HOST = TVCConstants.VOD_SERVER_INTL_HOST;
            TVCConstants.VOD_SERVER_HOST_BAK = TVCConstants.VOD_SERVER_INTL_HOST_BAK;
            UGCReport.REPORT_URL = TVCConstants.VOD_REPORT_INTL;
            UGCReport.REPORT_URL_BAK = TVCConstants.VOD_REPORT_INTL_BAK;
        } else {
            TVCLog.e(TAG, "unknown env type:" + envType);
        }
    }
}
